/*
 * Decompiled with CFR 0.152.
 */
package org.minidns.dane;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.security.KeyManagementException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import javax.security.cert.CertificateEncodingException;
import javax.security.cert.X509Certificate;
import org.minidns.AbstractDnsClient;
import org.minidns.dane.DaneCertificateException;
import org.minidns.dane.ExpectingTrustManager;
import org.minidns.dnsmessage.DnsMessage;
import org.minidns.dnsname.DnsName;
import org.minidns.dnssec.DnssecClient;
import org.minidns.dnssec.DnssecMessage;
import org.minidns.dnssec.UnverifiedReason;
import org.minidns.record.Record;
import org.minidns.record.TLSA;

public class DaneVerifier {
    private static final Logger LOGGER = Logger.getLogger(DaneVerifier.class.getName());
    private final AbstractDnsClient client;

    public DaneVerifier() {
        this((AbstractDnsClient)new DnssecClient());
    }

    public DaneVerifier(AbstractDnsClient client) {
        this.client = client;
    }

    public boolean verify(SSLSocket socket) throws CertificateException {
        if (!socket.isConnected()) {
            throw new IllegalStateException("Socket not yet connected.");
        }
        return this.verify(socket.getSession());
    }

    public boolean verify(SSLSession session) throws CertificateException {
        try {
            return this.verifyCertificateChain(DaneVerifier.convert(session.getPeerCertificateChain()), session.getPeerHost(), session.getPeerPort());
        }
        catch (SSLPeerUnverifiedException e) {
            throw new CertificateException("Peer not verified", e);
        }
    }

    public boolean verifyCertificateChain(java.security.cert.X509Certificate[] chain, String hostName, int port) throws CertificateException {
        DnsMessage res;
        DnsName req = DnsName.from((String)("_" + port + "._tcp." + hostName));
        try {
            res = this.client.query(req, Record.TYPE.TLSA);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        if (!res.authenticData) {
            String msg = "Got TLSA response from DNS server, but was not signed properly.";
            if (res instanceof DnssecMessage) {
                msg = msg + " Reasons:";
                for (UnverifiedReason reason : ((DnssecMessage)res).getUnverifiedReasons()) {
                    msg = msg + " " + reason;
                }
            }
            LOGGER.info(msg);
            return false;
        }
        LinkedList<DaneCertificateException.CertificateMismatch> certificateMismatchExceptions = new LinkedList<DaneCertificateException.CertificateMismatch>();
        boolean verified = false;
        for (Record record : res.answerSection) {
            if (record.type != Record.TYPE.TLSA || !record.name.equals((Object)req)) continue;
            TLSA tlsa = (TLSA)record.payloadData;
            try {
                verified |= DaneVerifier.checkCertificateMatches(chain[0], tlsa, hostName);
            }
            catch (DaneCertificateException.CertificateMismatch certificateMismatchException) {
                certificateMismatchExceptions.add(certificateMismatchException);
            }
            if (!verified) continue;
            break;
        }
        if (!verified && !certificateMismatchExceptions.isEmpty()) {
            throw new DaneCertificateException.MultipleCertificateMismatchExceptions(certificateMismatchExceptions);
        }
        return verified;
    }

    private static boolean checkCertificateMatches(java.security.cert.X509Certificate cert, TLSA tlsa, String hostName) throws CertificateException {
        if (tlsa.certUsage == null) {
            LOGGER.warning("TLSA certificate usage byte " + tlsa.certUsageByte + " is not supported while verifying " + hostName);
            return false;
        }
        switch (tlsa.certUsage) {
            case serviceCertificateConstraint: 
            case domainIssuedCertificate: {
                break;
            }
            default: {
                LOGGER.warning("TLSA certificate usage " + tlsa.certUsage + " (" + tlsa.certUsageByte + ") not supported while verifying " + hostName);
                return false;
            }
        }
        if (tlsa.selector == null) {
            LOGGER.warning("TLSA selector byte " + tlsa.selectorByte + " is not supported while verifying " + hostName);
            return false;
        }
        byte[] comp = null;
        switch (tlsa.selector) {
            case fullCertificate: {
                comp = cert.getEncoded();
                break;
            }
            case subjectPublicKeyInfo: {
                comp = cert.getPublicKey().getEncoded();
                break;
            }
            default: {
                LOGGER.warning("TLSA selector " + tlsa.selector + " (" + tlsa.selectorByte + ") not supported while verifying " + hostName);
                return false;
            }
        }
        if (tlsa.matchingType == null) {
            LOGGER.warning("TLSA matching type byte " + tlsa.matchingTypeByte + " is not supported while verifying " + hostName);
            return false;
        }
        switch (tlsa.matchingType) {
            case noHash: {
                break;
            }
            case sha256: {
                try {
                    comp = MessageDigest.getInstance("SHA-256").digest(comp);
                    break;
                }
                catch (NoSuchAlgorithmException e) {
                    throw new CertificateException("Verification using TLSA failed: could not SHA-256 for matching", e);
                }
            }
            case sha512: {
                try {
                    comp = MessageDigest.getInstance("SHA-512").digest(comp);
                    break;
                }
                catch (NoSuchAlgorithmException e) {
                    throw new CertificateException("Verification using TLSA failed: could not SHA-512 for matching", e);
                }
            }
            default: {
                LOGGER.warning("TLSA matching type " + tlsa.matchingType + " not supported while verifying " + hostName);
                return false;
            }
        }
        boolean matches = tlsa.certificateAssociationEquals(comp);
        if (!matches) {
            throw new DaneCertificateException.CertificateMismatch(tlsa, comp);
        }
        return tlsa.certUsage == TLSA.CertUsage.domainIssuedCertificate;
    }

    public HttpsURLConnection verifiedConnect(HttpsURLConnection conn) throws IOException, CertificateException {
        return this.verifiedConnect(conn, null);
    }

    public HttpsURLConnection verifiedConnect(HttpsURLConnection conn, X509TrustManager trustManager) throws IOException, CertificateException {
        try {
            SSLContext context = SSLContext.getInstance("TLS");
            ExpectingTrustManager expectingTrustManager = new ExpectingTrustManager(trustManager);
            context.init(null, new TrustManager[]{expectingTrustManager}, null);
            conn.setSSLSocketFactory(context.getSocketFactory());
            conn.connect();
            boolean fullyVerified = this.verifyCertificateChain(DaneVerifier.convert(conn.getServerCertificates()), conn.getURL().getHost(), conn.getURL().getPort() < 0 ? conn.getURL().getDefaultPort() : conn.getURL().getPort());
            if (!fullyVerified && expectingTrustManager.hasException()) {
                throw new IOException("Peer verification failed using PKIX", expectingTrustManager.getException());
            }
            return conn;
        }
        catch (KeyManagementException | NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }

    private static java.security.cert.X509Certificate[] convert(Certificate[] certificates) {
        ArrayList<java.security.cert.X509Certificate> certs = new ArrayList<java.security.cert.X509Certificate>();
        for (Certificate certificate : certificates) {
            if (!(certificate instanceof java.security.cert.X509Certificate)) continue;
            certs.add((java.security.cert.X509Certificate)certificate);
        }
        return certs.toArray(new java.security.cert.X509Certificate[certs.size()]);
    }

    private static java.security.cert.X509Certificate[] convert(X509Certificate[] certificates) {
        java.security.cert.X509Certificate[] certs = new java.security.cert.X509Certificate[certificates.length];
        for (int i = 0; i < certificates.length; ++i) {
            try {
                certs[i] = (java.security.cert.X509Certificate)CertificateFactory.getInstance("X.509").generateCertificate(new ByteArrayInputStream(certificates[i].getEncoded()));
                continue;
            }
            catch (CertificateException | CertificateEncodingException e) {
                LOGGER.log(Level.WARNING, "Could not convert", e);
            }
        }
        return certs;
    }
}

