const connection = require('./connection');
const getSpaces = require('./get-spaces').getSpaces;
const getArticles = require('./get-articles').getArticles;
const brokenApiFallbackGetArticles = require('./get-articles').brokenApiFallbackGetArticles;

const defaultState = {
    articlesSkip: 0
};
Object.freeze(defaultState);

function ConfluenceClient(context) {
    let state = Object.assign({}, defaultState);
    const http = connection.makeHttpConnection(context);
    const loadJson = connection.loadJsonFactory(http);
    const postJson = connection.postJsonFactory(http);

    function getServerInfo() {
        const result = http.getSync('/rest/applinks/1.0/manifest');
        const serverTimeHeader = result.headers.filter(h => h.name.toLowerCase() === 'date')[0];
        const resJson  = JSON.parse(result.response);
        return {
            version: resJson.version,
            time: new Date(serverTimeHeader.value).toISOString()
        };
    }

    function getUsers() {
        return [];
    }

    function getAttachmentContent(project, issue, attachment) {
        const res = loadJson('rest/api/content/' + attachment.id);

        const response = http.getSync(res._links.download);
        return {
            data: response.responseAsStream,
            metadata: {
                mimeType: res.metadata.mediaType
            }
        };
    }

    function getArticlesAndBumpPosition(project, after, top) {
        const serverInfo = getServerInfo();
        let articles;
        if (serverInfo.version.indexOf('5.') === 0) {
            console.log('Server version is ' + serverInfo.version + ', using brokenApiFallbackGetArticles');
            articles = brokenApiFallbackGetArticles(loadJson, postJson, project, state.articlesSkip, top);
        } else {
            articles = getArticles(loadJson, project, state.articlesSkip, top);
        }

        state.articlesSkip += articles.length;
        console.log('>>>>>>> Loaded articles', articles.length, ', articlesSkip=', state.articlesSkip)

        return articles;
    }



    return {
        prepareToImport: function() {
            state = Object.assign({}, defaultState);
        },
        getServerInfo: getServerInfo,
        getProjects: () => getSpaces(loadJson),
        getUsers: getUsers,
        getAttachmentContent: getAttachmentContent,
        getArticles: getArticlesAndBumpPosition,
        getTimestampFormats: function () {
            return ["yyyy-MM-dd'T'HH:mm:ss.SSSX", "yyyy-MM-dd'T'HH:mm:ssX", "yyyy-MM-dd'T'HH:mmX", "yyyy-MM-dd"];
        }
    };
}

exports.Client = ConfluenceClient;