const toMarkdown = require('./to-markdown').toMarkdown;

const EXPAND = 'ancestors,body.storage,history,history.lastUpdated,children.comment.body.storage,children.comment.history.lastUpdated,children.attachment,children.attachment.history';

function makeUser(user) {
  return {
    id: user.username, // TODO: proper ID
    fullName: user.displayName,
    name: user.displayName,
    login: user.username,
    type: 'user'
  };
}


function makeGetters(loadJson, postJson, useDeprecatedAPI) {
  if (useDeprecatedAPI) {
    return {
      getUser: function deprecatedGetUser(userKey) {
        try {
          const res = postJson('rpc/json-rpc/confluenceservice-v2', {os_authType: 'basic'},
            JSON.stringify({jsonrpc: '2.0', method: 'getUserByKey', params: [userKey], id: 1})
          );

          return res.result.name;
        } catch (e) {
          console.warn('Failed to retrieve user details for ID=' + userKey);
          return userKey;
        }
      }
    };
  }

  return {
    getUser: (userKey) => {
      try {
        const user = loadJson('rest/api/user', {key: userKey});
        return user.username;
      } catch (e) {
        console.warn('Failed to retrieve user details for ID=' + userKey);
        return userKey;
      }
    }
  }
}


function convertPageToArticle(loadJson, page, getters) {
  // const parent = page.ancestors[page.ancestors.length - 1];

  try {
    return {
      id: page.id.toString(),
      key: page.id.toString(),
      fields: {
        summary: page.title,
        content: toMarkdown(page.body.storage.value, getters),
        created: page.history.createdDate,
        updated: page.history.lastUpdated.when,
        author: makeUser(page.history.createdBy),
        // links: parent ? [{
        //     linkName: 'subtask of',
        //     issueId: parent.id.toString()
        // }] : [],
        attachments: page.children.attachment.results.map(attachment => ({
          id: attachment.id,
          filename: attachment.title,
          created: attachment.history.createdDate,
          author: makeUser(attachment.history.createdBy),
          mimeType: attachment.metadata.mediaType
        })),
        comments: page.children.comment.results.map(comment => ({
          id: comment.id,
          text: toMarkdown(comment.body.storage.value, getters),
          author: makeUser(comment.history.createdBy),
          updated: comment.history.lastUpdated.when,
          created: comment.history.createdDate,
        }))
      }
    };
  } catch (e) {
    console.log('Failed to convert pageId="' + page.id + '"');
    console.log('>>>>>>>>>>Begin of page content>>>>>>>>>>\n\n' + JSON.stringify(page) + '\n\n<<<<<<<<<<<End of page content<<<<<<<<<<<')
    console.log('>>>> Original error stack:', e.stack);
    throw e;
  }
}

function getArticlesPiece(loadJson, spaceKey, start, limit) {
  console.trace('getArticlesPiece called start=', start, 'limit=', limit);

  // https://developer.atlassian.com/cloud/confluence/rest/#api-api-content-get
  const pagesRes = loadJson('rest/api/content', {
    spaceKey: spaceKey,
    start: start.toString(),
    limit: limit.toString(),
    expand: EXPAND
  });

  console.trace('getArticlesPiece loaded ' + pagesRes.size + ' pages for space "' + spaceKey + '"')

  return pagesRes.results.map(page => convertPageToArticle(loadJson, page, makeGetters(loadJson, null, false)));
}


// On confluence 5.x version REST API pagination is broken (ADM-42470)
function brokenApiFallbackGetArticles(loadJson, postJson, project, start, limit) {
  // See https://developer.atlassian.com/server/confluence/remote-confluence-methods/#pages
  const allPages = postJson('rpc/json-rpc/confluenceservice-v2',
    {os_authType: 'basic'},
    JSON.stringify({jsonrpc : '2.0', method : 'getPages', params : [project.id], id: 1})
  );

  const allIds = allPages.result.map(r => r.id);
  const pageIds = allIds.slice(start, start + limit);
  console.trace('brokenApiFallbackGetArticles is about to load ' + pageIds.join(',') + ' pages for space "' + project.id + '"');

  const pages = pageIds.map(id => {
    return loadJson('rest/api/content/' + id, {expand: EXPAND});
  });

  console.trace('brokenApiFallbackGetArticles loaded ' + pages.length + ' pages for space "' + project.id + '"')

  return pages.map(page => convertPageToArticle(loadJson, page, makeGetters(loadJson, postJson, true)));
}

const MAX_PAGES = 10;

function getArticles(loadJson, project, skip, top) {
  let pageNum = 0;
  let from = 0;
  let done = false;
  let result = [];

  while (!done && ++pageNum <= MAX_PAGES) {
    let articles = getArticlesPiece(loadJson, project.id, skip + from, top - from)
    result = result.concat(articles);

    from += articles.length;
    done = result.length >= top;
  }

  return result;
}

exports.getArticles = getArticles;
exports.brokenApiFallbackGetArticles = brokenApiFallbackGetArticles;