const dependencies = require('./third-party-dependencies');

const toText = dependencies.utils.toText;
const all = dependencies.utils.all;

function cleanPlainTextBody(plainTextBodyNode, wholeContent) {
  const originalContent = wholeContent.substring(
    plainTextBodyNode.position.start.offset,
    plainTextBodyNode.position.end.offset
  );
  return originalContent
    .replace('<ac:plain-text-body><![CDATA[', '')
    .replace(']]></ac:plain-text-body>', '')
    .replace(/^\n/, '')
    .replace(/\n$/, '');
}

function handleCodeMacro(content, h, node) {
  const langNode = node.children.filter(c => c.properties['ac:name'] === 'language')[0];
  const lang = langNode ? langNode.children[0].value : null;

  const code = node.children.filter(c => c.tagName === 'ac:plain-text-body')[0];

  return h(node, 'code', {lang: lang, meta: null}, cleanPlainTextBody(code, content));
}

function handlePanel(content, h, node) {
  // const title = node.children.filter(c => c.properties['ac:name'] === 'title')[0];

  const richText = node.children.filter(c => c.tagName === 'ac:rich-text-body')[0];
  const codeNodes = richText.children.filter(c => c.properties['ac:name'] === 'unmigrated-inline-wiki-markup');
  const code = codeNodes.map(n => cleanPlainTextBody(n.children[0], content))
    .join('\n')
    .replace(new RegExp('{pre}', 'g'), '')
    .replace(/\\{/g, '{')
    .replace(/\\}/g, '}')
    .replace(/\\\[/g, '[')
    .replace(/\\\]/g, ']');

  return h(node, 'code', {lang: null, meta: null}, code);
}

function handleJiraMacro(content, h, node) {
  const jqlQuery = node.children.filter(c => c.properties['ac:name'] === 'jqlQuery')[0];

  return h(node, 'inlineCode', jqlQuery ? toText(jqlQuery) : '');
}

function handleJiraIssuesMacro(content, h, node) {
  const urlParam = node.children.filter(c => c.properties['ac:name'] === 'url')[0];
  const urlNode = urlParam.children[0];
  const src = urlNode.properties['ri:value'];

  return h(node, 'link', {url: src}, [h(node, 'text', src)]);
}

function handleUnmigratedWikiMarkup(content, h, node) {
  const cleaned = cleanPlainTextBody(node.children[0], content);
  return h(node, 'code', {lang: null, meta: null}, cleaned);
}

const macroHandlers = {
  code: handleCodeMacro,
  panel: handlePanel,
  jira: handleJiraMacro,
  jiraissues: handleJiraIssuesMacro,
  'unmigrated-inline-wiki-markup': handleUnmigratedWikiMarkup
};

function handleMacros(content, h, node) {
  const name = node.properties['ac:name'];

  try {
    if (macroHandlers[name]) {
      return macroHandlers[name](content, h, node);
    }

    console.trace('WARNING: Unknown confluence macros', name, 'passing content through.');
    return all(h, node);
  } catch (err) {
    console.error('Failed to transform macros', name, 'passing content through. Error: ', err);
    return all(h, node);
  }
}

module.exports = {
  cleanPlainTextBody: cleanPlainTextBody,
  handleMacros: handleMacros
};
