const dependencies = require('./third-party-dependencies');
const macroHandlers = require('./macro-handlers');

const unified = dependencies.unified;
const parse = dependencies.parse;
const rehype2remark = dependencies.rehype2remark;
const stringify = dependencies.stringify;

const all = dependencies.utils.all;
const handlers = dependencies.utils.handlers;
const wrapText = dependencies.utils.wrapText;
const toText = dependencies.utils.toText;

function cleanCData(text) {
  return text.replace(/^\[CDATA\[/, '').replace(/\]\]$/, '');
}

module.exports = {
  toMarkdown: function(content, getters) {
    const getUser = getters.getUser;

    function handleMacros(h, node) {
      return macroHandlers.handleMacros(content, h, node);
    }

    const processor = unified()
      .use(parse, {fragment: true})
      .use(rehype2remark, {
        handlers: {
          span: (h, node) => {
            const style = node.properties.style || '';
            if (style.indexOf('text-decoration: line-through') >= 0) {
              return handlers.s(h, node);
            }
            return all(h, node);
          },
          'ri:page': (h, node) => {
            const pageName = node.properties['ri:content-title'];
            // TODO: find a way to get article idReadable and use it instead

            const text = h(node, 'text', pageName);
            return h(node, 'link', {url: pageName}, [text])
          },
          'ri:user': (h, node) => {
            const user = getUser(node.properties['ri:userkey']);
            return h(node, 'text', wrapText(h, ' @' + user));
          },
          'ac:image': (h, node) => {
            const content = node.children[0];
            if (!content) {
              return all(h, node);
            }
            const imageSrc = content.tagName === 'ri:url' ? content.properties['ri:value'] : content.properties['ri:filename'];

            return h(node, 'image', {url: imageSrc});
          },
          'ac:structured-macro': handleMacros,
          'ac:macro': handleMacros,
          'ac:plain-text-body': (h, node) => {
            const cleaned = macroHandlers.cleanPlainTextBody(node, content);
            return h(node, 'text', cleaned);
          },
          'ac:task-list': (h, node) => {
            return handlers.ol(h, node);
          },
          'ac:task': (h, node) => {
            const status = node.children.filter(c => c.tagName === 'ac:task-status')[0];
            const checked = toText(status) === 'complete';
            const content = node.children.filter(c => c.tagName === 'ac:task-body')[0] || [];

            return h(
              node,
              'listItem',
              {spread: false, checked: checked},
              all(h, content)
            )
          },
          comment: (h, node) => {
            return h(node, 'text', wrapText(h, cleanCData(node.value)));
          }
        }
      })
      .use(stringify, {
        commonmark: false,
        gfm: true,
        bullet: '*',
        listItemIndent: '1',
        incrementListMarker: true,
        emphasis: '*',
        tablePipeAlign: false,
        fences: true
      });

    const res = processor.processSync(content);

    return res.toString();
  }
}