const http = require('@jetbrains/youtrack-scripting-api/http');

GitHubIssues = function (context) {

  const getTimestampFormats = () => {
    return ["yyyy-MM-dd'T'HH:mm:ss'Z'", "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", "EEE, d MMM yyyy HH:mm:ss z"];
  };

  const imageAttachmentRegexp = /(?:!\[([^[]+)\]\((https:\/\/user-images\.githubusercontent\.com\/\d*\/([^)]*))\))/mg;
  const fileAttachmentRegexp = /\[([^)]+)\]\((https:\/\/github\.com\/[\d\w-_+]*\/[\d\w-_+]*\/files\/\d*\/([^)]*))\)/mg;

  const params = context.parameters;
  let url = params.loadValue('url');
  const sslKeyName = params.loadValue('sslKeyName');
  const token = params.loadValue('password');
  const connection = new http.Connection(url, sslKeyName)
    .addHeader('Authorization', 'token ' + token)
    .addHeader('Accept', 'application/vnd.github.v3+json');
  const fileConnection = new http.Connection('', sslKeyName);

  const client = {};

  const getProjects = () => {
    const ghProjects = loadJson('/user/repos');
    return ghProjects.map(project => {
      return {
        id: project.id,
        name: project.name,
        key: project.name,
      }
    });
  };

  const getProject = (projectKey, failureHandler) => {
    const project = loadJson('/repos/' + projectKey.id);
    //GET /repos/:owner/:repo/milestones
    const milestones = loadJson('/repos/' + projectKey.id + '/milestones', {'state': 'all'});
    const convertedMilestones = milestones.map(function (milestone) {
      return {
        name: milestone.title,
        description: milestone.description,
        releaseDate: milestone.due_on,
        released: milestone.state === 'closed'
      }
    });
    this.projectOwner = project.owner;
    return {
      id: projectKey.id,
      name: project.name,
      key: projectKey.key,
      lead: {
        fullName: project.owner.login,
        name: project.owner.login
      },
      fields: [
        {
          id: 'State',
          name: 'State',
          multiValue: false,
          type: 'state',
          values: [
            {
              name: 'Open',
              isResolved: false
            },
            {
              name: 'Closed',
              isResolved: true
            }
          ]
        },
        {
          id: 'Type',
          name: 'Type',
          multiValue: true,
          type: 'enum',
          values: [
            {
              name: 'bug',
              description: 'Something isn\'t working'
            },
            {
              name: 'enhancement',
              description: 'New feature or request'
            },
            {
              name: 'question',
              description: 'Further information is requested'
            }
          ]
        },
        {
          id: 'Milestone',
          name: 'Milestone',
          multiValue: true,
          type: 'version',
          values: convertedMilestones
        }
      ]
    }
  };

  const prepareToImport = () => {
    console.log('Preparing to import issues from GitHub server with baseUrl ' + url);
    this.attachmentsMap = {};
  };

  const headersToMap = (headers) => {
    const result = {};
    headers.forEach(header => {
      result[header.name.toLowerCase()] = header.value;
    });
    return result;
  };

  const getServerInfo = () => {
    const response = connection.getSync('/?');
    const headers = headersToMap(response.headers);
    const versionHeader = headers['x-github-media-type'];
    return {
      version: versionHeader.split(';')[0],
      time: headers['date']
    };
  };

  const getRepoFullName = (repoId) => {
    return loadJson(repoId).full_name;
  };

  const loadJson = (uri, queryParams) => {
    const result = connection.getSync(uri, queryParams);
    if (!result.isSuccess) {
      throw 'Failed to load data from ' + uri + ' code ' + result.code;
    } else {
      return JSON.parse(result.response);
    }
  };

  const generateProjectKey = (key) => {
    key.split(/[\s,-.]+/).filter(word => word.length).map(word => word.substring(0, 1).toUpperCase())
  };


  const getIssues = (project, from, top) => {
    let count = 0;
    let ghIssues = [];
    let updated = null;
    const query = {
      'sort': 'updated',
      'direction': 'asc',
      'state': 'all'
    };
    const loadedIssues = {};

    const issuesWithComments = {};

    const attachmentsMap = this.attachmentsMap;

    while (count < top) {
      let hasNewIssues = false;
      if (updated !== null) {
        query.since = updated.toString();
      }
      let issues = loadJson('/repos/' + project.id + '/issues', query);
      if (issues.length) {
        if (updated === null) {
          updated = issues[0].updated_at;
        }
        issues.forEach(function (issue) {
          if (updated < issue.updated_at) {
            updated = issue.updated_at;
          }
          if (!loadedIssues[issue.id]) {
            issue.attachments = [];
            loadedIssues[issue.id] = true;
            if (issue.pull_request == null) {
              ghIssues.push(issue);
              count++;
              hasNewIssues = true;
              if (issue.comments) {
                issue.gh_comments = loadIssueComments(project.id, issue.number, issue.attachments, attachmentsMap);
                issue.gh_comments.forEach(function (comment) {
                  extractAttachmentsFromComment(issue, comment, attachmentsMap);
                })
              }
            }
          }
        });
      } else {
        break;
      }
      if (!hasNewIssues) {
        break;
      }
    }
    console.log('Loaded ' + count + ' issues ');
    ghIssues = ghIssues.map(convertGHIssue);
    return ghIssues;
  };

  const processMarkdownImageRegexp = function (issue, comment, attachmentsMap, name, link, ghFilename) {
    if (!issue.attachments) {
      issue.attachments = [];
    }
    const attachment = {};
    issue.attachments.push(attachment);
    attachment.author = getCommentAuthor(comment);
    attachment.id = comment.id + '-' + ghFilename;
    attachment.created = comment.created_at;
    attachment.mimeType = 'unknown';
    attachment.charset = 'UTF-8';
    attachmentsMap[attachment.id] = link;
    return attachment;
  };

  const extractAttachmentsFromComment = (issue, comment, attachmentsMap) => {
    comment.body = comment.body.replace(imageAttachmentRegexp, function (match, name, link, ghFilename) {
      const attachment = processMarkdownImageRegexp(issue, comment, attachmentsMap, name, link, ghFilename);
      const extension = (ghFilename.lastIndexOf('.') !== -1 && ghFilename.substring(ghFilename.lastIndexOf(ghFilename.lastIndexOf('.')))) || '';
      attachment.filename = name + extension;
      return '![' + name + '](' + name + extension + ')'
    });
    comment.body = comment.body.replace(fileAttachmentRegexp, function (match, name, link, ghFilename) {
      const attachment = processMarkdownImageRegexp(issue, comment, attachmentsMap, name, link, ghFilename);
      attachment.filename = name;
      return '[' + name + '](' + name + ')'
    });
  };

  const getAttachmentContent = function (project, issue, attachment) {
    const link = this.attachmentsMap[attachment.id];
    const response = fileConnection.getSync(link);
    const headers = headersToMap(response.headers);
    const mimeType = headers['content-type'];
    return {
      data: response.responseAsStream,
      metadata: {
        mimeType: mimeType
      }
    };
  };

  const loadIssueComments = function (projectId, issueId) {
    return loadJson('/repos/' + projectId + '/issues/' + issueId + '/comments');
  };

  const labelIsType = function (label) {
    return label.name === 'enhancement' || label.name === 'bug' || label.name === 'question';
  };

  const getCommentAuthor = (comment) => {
    return {
      id: comment.user.id,
      fullName: comment.user.login,
      name: comment.user.login,
      type: 'user'
    }
  };

  const convertGHComment = function (comment) {
    return {
      id: comment.id + '',
      text: comment.body,
      author: getCommentAuthor(comment),
      updated: comment.updated_at,
      created: comment.created_at
    };
  };

  const convertGHIssue = function (issue) {
    return {
      id: issue.number,
      key: issue.number + '',
      fields: {
        summary: issue.title,
        description: issue.body,
        created: issue.created_at,
        updated: issue.updated_at,
        author: {
          id: issue.user.id,
          fullName: issue.user.login,
          name: issue.user.login,
          type: 'user'
        },
        assignee: issue.assignee && {
          id: issue.assignee.id,
          fullName: issue.assignee.login,
          name: issue.assignee.login,
          type: 'user'
        },
        attachments: issue.attachments,
        comments: issue.gh_comments && issue.gh_comments.map(convertGHComment),
        tags: issue.labels.filter(function (label) {
          return !labelIsType(label)
        }).map(function (label) {
          return {
            id: label.id,
            name: label.name,
            type: 'tag',
            visibleToGroups: [
              'All Users'
            ]
          }
        }),
        state: issue.state,
        type: issue.labels.filter(labelIsType).map(function (label) {
          return label.name
        }),
        resolved: issue.closed_at
      }
    };
  };

  const getUsers = function () {
    return [];
  };

  Object.assign(client, {
    getProjects: getProjects.bind(this), // api
    getProject: getProject.bind(this), // api
    getLinkTypes: () => [],
    getServerInfo: getServerInfo.bind(this), // api
    getAttachmentContent: getAttachmentContent.bind(this),
    getIssues: getIssues.bind(this), // api
    getUsers: getUsers.bind(this),
    prepareToImport: prepareToImport.bind(this), // api
    getTimestampFormats: getTimestampFormats.bind(this) // api
  });
  return client;
};

exports.Client = GitHubIssues;
