/*
 Copyright 2017 JetBrains s.r.o.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

const dateTime = require('@jetbrains/youtrack-scripting-api/date-time');
const http = require('@jetbrains/youtrack-scripting-api/http');
const cache = require('@jetbrains/youtrack-scripting-api/cache');

const IssueHandler = require('./issue-handler');
const Events = require('./events');
const Fields = require('./fields');
const Security = require('./security');

const jiraLinks = require('./links');
const jiraUtils = require('./utils');
const projectFieldPropertyNames = ['id', 'name', 'type', 'multiValue'];

const usersDirectory = require('./usersDirectory');

const createFakeSeraphNotOkResponse = function (result) { // required to handle https://jira.atlassian.com/browse/JRA-41559
  const seraphHeaders = result.headers && result.headers.filter(header => header.name === 'X-Seraph-LoginReason') || [];
  const seraphNotOkHeaders = seraphHeaders.filter(header => header.value.indexOf('Failure') > -1);
  if (seraphHeaders.length > 0 && seraphNotOkHeaders.length > 0) {
    return {
      response: '{"errorMessages": "Response X-Seraph-LoginReason header set to ' + seraphNotOkHeaders[0].value + '. You are probably not authenticated"}',
      code: result.code
    }
  }
};

const toSearchTimeString = (timestamp, timeZoneId) => timestamp && dateTime.format(timestamp, 'yyyy-MM-dd HH:mm', timeZoneId);

// -------------------- failure handlers ----------------

const createCommonFailureHandler = (exceptionHandlingFunction, errorMessagePrefix) => {
  return (failure) => {
    if (failure.exception) {
      exceptionHandlingFunction(failure.exception);
      return;
    }
    let errorJson = null;
    if (failure.response) {
      try {
        errorJson = JSON.parse(failure.response);
      } catch (e) {
        // ignore, html can be here, not json
      }
    }
    exceptionHandlingFunction(errorMessagePrefix +
      (errorJson ? (failure.code + ' ' + errorJson.errorMessages) : ('Server replied with status code ' + failure.code))
    );
  };
};

const networkFailureHandler = createCommonFailureHandler((error) => { throw error }, '');

const attachmentContentFailureHandler = createCommonFailureHandler(console.error, 'Error when requesting attachment content. ');

const composeFailureHandler = function(customFailureHandler) {
  return result => {
    console.error('Error occurred when querying Jira');
    if (result.response) {
      let errorJson = null;
      try {
        errorJson = JSON.parse(result.response);
      } catch (e) {
        // ignore, html can be here, not json
      }
      console.error(result.code + ' ' + (errorJson ? errorJson.errorMessages : ''));
    }
    result.exception && console.error(result.exception);
    customFailureHandler && customFailureHandler(result);
  };
};
// -------------------- end of failure handlers ----------------

Jira = function (context) {
  const params = context.parameters;
  let url = params.loadValue('url');
  const sslKeyName = params.loadValue('sslKeyName');
  const login = params.loadValue('login');
  const password = params.loadValue('password');
  url = url + (url.lastIndexOf('/') === url.length - 1 ? '' : '/');
  const connection = new http.Connection(url + 'rest/api/2', sslKeyName).basicAuth(login, password);
  const authConnection = new http.Connection(url + 'rest/auth/1', sslKeyName).basicAuth(login, password);
  const plainConnection = new http.Connection(url + 'secure', sslKeyName).basicAuth(login, password);
  const client = {};

  // ----------------- network routine ------------------
  const getJsonsCache = function () {
    return jiraUtils.getOrCreateObject(context, 'jsonsCache', () => {
      return cache.create(100, 'JSONs');
    });
  };

  const getSkippingFailureHandler = (statusCodesToSkip) => {
    return (failure) => {
      failure.code && (statusCodesToSkip.indexOf(failure.code) !== -1) || networkFailureHandler(failure);
    }
  };

  const doGetJson = (connection, uri, failureHandler, ignoreCache) => {
    failureHandler = failureHandler ? failureHandler : networkFailureHandler;
    const doIt = function () {
      const result = connection.getSync(uri);
      let fakeResponse;
      if (!result.isSuccess || (fakeResponse = createFakeSeraphNotOkResponse(result))) {
        composeFailureHandler(failureHandler)(fakeResponse || result);
        return;
      }
      return JSON.parse(result.response);
    };
    return ignoreCache ? doIt() : getJsonsCache().get(uri, doIt);
  };

  const getJson = (uri, failureHandler, ignoreCache) => {
    return doGetJson(connection, uri, failureHandler, ignoreCache);
  };

  const getPaginated = (searchUri, arrayName, pageSize, successHandler, failureHandler) => {
    let startAt = 0;
    let actualPageSize = 0;
    let result = null;
    do {
      let uri = searchUri + (searchUri.indexOf('?') > -1 ? '&' : '?') + 'startAt=' + startAt + '&maxResults=' + pageSize;
      let callResult = getJson(uri, failureHandler, true);
      result = arrayName ? callResult[arrayName] : callResult;
      actualPageSize = result ? result.length : 0;
      startAt += actualPageSize;
    } while (successHandler(result) && actualPageSize === pageSize);
  };

  // ----------------- end of network routine ------------------

  const getIssueChangelog = (issueKey, failureHandler) => {
    // there's not need to get any field info here, but there is no way to turn it off completely
    return getJson('/issue/' + issueKey + '?fields=created&expand=changelog', failureHandler);
  };

  const getProjectPermissions = (projectKey, failureHandler) => {
    const ret = getJson('/mypermissions?permissions=BROWSE_PROJECTS,CREATE_ISSUES,ADMINISTER_PROJECTS&projectKey=' + projectKey, failureHandler);
    const original = ret && ret['permissions'];
    const permissions = {};
    Object.keys(jiraUtils.permissionKeys).forEach(function (permissionKey) {
      permissions[permissionKey] = jiraUtils.permissionKeys[permissionKey].some(function (alias) {
        return original && original[alias] && original[alias].havePermission;
      });
    });
    return permissions;
  };

  const accept = (field, condition, reason) => {
    if (!condition) {
      console.trace('Ignoring field ' + field.id + ' (' + field.name + ') as ' + reason);
    }
    return condition;
  };

  const acceptField = (field, issueHandler, fieldId) => {
    return accept(field, field.schema, 'as it does not have \'schema\' attribute') &&
        accept(field, issueHandler.unsupportedFieldTypes.indexOf(field.schema.custom) === -1, 'it is of an unsupported type ' + field.schema.custom) &&
        accept(field, issueHandler.projectFieldsToIgnore.indexOf(fieldId) === -1, 'it is a predefined field not a field in YouTrack') &&
        accept(field, issueHandler.issueFieldsToIgnore.indexOf(fieldId) === -1, 'it will be ignored when processing issues');
  };

  const getProject = (projectInfo) => {
    const projectKey = projectInfo.key;
    const projectPermissions = getProjectPermissions(projectKey, function (result) {
      if (result.code === 404) {
        console.warn('Project is not accessible: ' + projectKey);
        networkFailureHandler('Project is not accessible: ' + projectKey);
      } else {
        networkFailureHandler(result);
      }
    });
    console.info(projectKey + ' permissions: ' +
      Object.keys(projectPermissions).map(key => key + ':' + projectPermissions[key]).join());
    if (!projectPermissions['BROWSE']) {
      console.warn('No permissions to view the project: ' + projectKey);
      networkFailureHandler('No permissions to view the project: ' + projectKey);
    }

    const foreignProject = getJson('/project/' + projectKey, networkFailureHandler);
    foreignProject.lead = client.$private.fields.user(foreignProject.lead)[0];
    foreignProject.$permissions = projectPermissions;
    foreignProject.$fieldValues = {};
    const fieldSchema = jiraUtils.getObject(context, "fieldSchema");
    fieldSchema.projects[projectKey] = foreignProject;

    const issueHandler = client.$private.issueHandler;
    const foreignProjectFields = getProjectFields(foreignProject);
    console.info('Transforming project fields of project ' + projectKey);
    foreignProject.fields = Object.keys(foreignProjectFields)
      .filter(fieldId => {
        const field = foreignProjectFields[fieldId];
        return acceptField(field, issueHandler, fieldId);
      })
      .map(fieldId => {
        const field = foreignProjectFields[fieldId];
        const fieldName = field.name;
        let existingField = fieldSchema.findField(fieldId, fieldName);
        if (existingField) {
          console.debug('Found field "' + existingField.name + '" that matches project field "' + fieldName + '" by id "' + fieldId + '"');
        } else {
          console.info('Adding field "' + fieldName + '" to schema based on the project field with id "' + fieldId + '"');
          issueHandler.defineFieldType(field);
          existingField = jiraUtils.copyProperties(field, {originalName: fieldName}, projectFieldPropertyNames);
          fieldSchema.storeField(fieldId, fieldName, existingField);
        }
        const fieldInfo = fieldId + ' (type: ' + existingField.type + ')';
        if (issueHandler.projectFieldTypesToIgnore.indexOf(existingField.type) > -1) {
          console.trace('Project ' + projectKey + '. Ignoring field ' + fieldInfo + ' as it\'s type is ' +
            'to be ignored when processing project fields');
          return null;
        }
        const converter = client.$private.fields[fieldId] || client.$private.fields[existingField.type];
        if (converter) {
          console.debug('Project ' + projectKey + '. Converting value of project field ' + fieldInfo);
          foreignProject.$fieldValues[fieldId] = field.allowedValues && converter(field.allowedValues);
          return jiraUtils.copyProperties(existingField, {values: foreignProject.$fieldValues[fieldId]}, projectFieldPropertyNames);
        } else {
          console.warn('Project ' + projectKey + '. No converter for field ' + fieldInfo + '. It will be skipped during issues processing.');
          return null;
        }
    }).filter(elem => elem); // filter nulls out
    console.info('Done transforming project fields of project ' + projectKey);
    return foreignProject;
  };

  const getProjectFields = (project) => {
    const projectFields = {
      'issuetype': {
        'id': 'issuetype',
        'name': 'Issue Type',
        'schema': {
          'type': 'issuetype',
          'system': 'issuetype'
        },
        'valuesMap': {}
      },
      'assignee': {
        'id': 'assignee',
        'name': 'Assignee',
        'schema': {
          'type': 'user',
          'system': 'assignee'
        },
        'valuesMap': {}
      }
    };
    const projectKey = project.key;
    if (!project.$permissions['CREATE_ISSUE']) {
      console.info('Can not retrieve project fields schema for ' + projectKey + ' project, no CREATE_ISSUE permission');
      return projectFields;
    }
    console.debug('Retrieving issue types for project ' + projectKey);
    const projectsMeta = getJson('/issue/createmeta?projectKeys=' + projectKey + '&expand=projects.issuetypes.fields.allowedValues', null, true);
    const issueTypes = projectsMeta.projects[0].issuetypes;
    issueTypes.forEach(function(issueType) {
      console.debug('Converting issue type ' + issueType.name);
      Object.keys(issueType.fields).forEach(function (fieldId) {
        const field = issueType.fields[fieldId];
        const projectField = projectFields[fieldId] ||
          jiraUtils.copyProperties(field, {'id': fieldId, 'multiValue': !!field.allowedValues}, ['schema', 'name']);
        if (!projectFields[fieldId]) {
          projectFields[fieldId] = projectField
        }
        if (field.allowedValues) {
          projectField.valuesMap = projectField.valuesMap || {};
          field.allowedValues.forEach(value => projectField.valuesMap[value.id] = value);
        }
      });
    });
    Object.keys(projectFields).forEach(projectFieldId => {
      const projectField = projectFields[projectFieldId];
      if (projectField.valuesMap) {
        projectField.allowedValues = Object.keys(projectField.valuesMap).map(key => projectField.valuesMap[key]);
      }
      console.debug('Project field ' + projectField.name + ' (id: ' + projectFieldId + '), allowedValues: '
        + JSON.stringify(projectField.allowedValues));
    });
    return projectFields;
  };

  const getStatuses = failureHandler => getJson('/status', failureHandler);

  const getRole = (projectKey, roleId) => {
    const ret = getJson('/project/' + projectKey + '/role/' + roleId, getSkippingFailureHandler([401]));
    if (ret) { // everything went fine
      return ret;
    }
    if (roleId) {
      return null;
    } else {
      return [];
    }
  }

  const getComponent = (componentId, failureHandler) => getJson('/component/' + componentId, failureHandler);

  const getIssue = (issueKey, failureHandler, compact) => {
    const json = getJson('/issue/' + issueKey + '?expand=changelog', failureHandler);
    return json && client.$private.issueHandler.fillIssue(json, failureHandler, compact);
  };

  const getIssues = (projectInfo, afterIssue, top) => {
    return getIssueUpdates(projectInfo, afterIssue, null, top)
  };

  const getIssueUpdates = (projectInfo, afterIssue, updatedAfter, top) => {
    const projectKey = projectInfo.key;
    const filters = [];
    const fromId = afterIssue && afterIssue.id;
    if (projectKey) {
      filters.push('project = "' + projectKey + '"');
    }
    if (updatedAfter) {
      filters.push('updatedDate >= "' + updatedAfter + '"');
    }
    if (fromId) {
      filters.push('id > ' + fromId);
    }

    const searchUri = '/search?jql=' + encodeURIComponent(filters.join(' and ') + ' order by id') + '&fields=*all&expand=changelog';

    let issues = [];
    getPaginated(searchUri, 'issues', top, jiraIssues => {
      jiraIssues.length && jiraIssues.forEach(jiraIssue => {
        client.$private.issueHandler.fillIssue(jiraIssue, null, false);
      });
      issues = jiraIssues;
      return false;
    });
    return issues;
  }

  const prepareFieldSchema = () => {
    const fieldSchema = jiraUtils.getOrCreateObject(context, 'fieldSchema', () => {
      return {
        statuses: null,
        resolutions: null,
        prototypes: {}, // keys are original jira ids
        nameToId: {}, // keys are original jira names
        projects: {},
        findField(id, name) {
          if (!id && !name) {
            throw "Neither id nor name of a field passed to findField";
          }
          const idByName = this.nameToId[name];
          return id && this.prototypes[id] || idByName && this.prototypes[idByName];
        },
        storeField(id, name, field) {
          this.prototypes[id] = field;
          this.nameToId[name] = id;
        }
      }
    });

    console.info('Loading field schema');
    const issueHandler = client.$private.issueHandler;
    const specialFields = issueHandler.specialFields;
    const fields = getJson('/field', networkFailureHandler);
    fields.filter(function (field) {
      // fields like issuekey and Images (thumbnails) do not have schema. Don't need such fields
      const fieldId = field.id;
      return acceptField(field, issueHandler, fieldId) &&
        accept(field, !specialFields[fieldId], 'it is a special field (like State, Assignee, etc)');
    }).forEach(function (field) {
      const id = field.id;
      const fieldName = field.name;
      console.debug('Redefining name and type of "' + fieldName + '"');
      issueHandler.defineFieldType(field);
      fieldSchema.storeField(id, fieldName, field);
    });
    Object.keys(specialFields).forEach((id) => {
      const specialField = specialFields[id];
      fieldSchema.storeField(id, specialField.originalName, specialField);
    });
    console.info('Field schema loaded');
  };

  const prepareToImport = () => {
    const sessionJson = doGetJson(authConnection, '/session', null, true);
    if (!sessionJson || !sessionJson.name) {
      throw 'Failed to connect to JIRA';
    }
    const isCloud = context.parameters.loadValue('isCloud') !== "false";
    jiraUtils.setObject(context, 'isCloud', isCloud);
    if (isCloud && !usersDirectory.init(context).initialized) {
      throw 'No Jira user data uploaded. Please upload export-users.csv that can be downloaded from ' + url + 'admin'
    }
    client.$private.sessionJson = sessionJson;
    client.prepareFieldSchema();
  };

  const getUsers = (jiraUserGroup, skip, top) => {
    const group = requestGroup(jiraUserGroup.name);
    const users = group && group.users && group.users.items || [];
    const convert = client.$private.fields.user;
    users.forEach(user => convert(user));
    return users;
  };

  const requestGroup = (groupName) => {
    return getJson('/group?groupname=' + encodeURIComponent(groupName) + '&expand=users', function (failure) {
      if (failure.code === 404) {
        console.warn('Could not find group ' + groupName + ', since Jira replied with status code 404');
      } if (failure.code === 403) {
        console.warn('Members of ' + groupName + ' won\'t be retrieved. Missing ADMINISTER permission');
      } else {
        networkFailureHandler(failure);
      }
    });
  };

  const getProjects = () => {
    const info = getJson('/serverInfo', networkFailureHandler, true);
    context.parameters.saveValue('isCloud', info.deploymentType === "Cloud");
    return getJson('/project');
  };

  const getLinkTypes = failureHandler =>
    jiraLinks.linkTypes(getJson('/issueLinkType', failureHandler).issueLinkTypes);

  const getTimestampFormats = () => {
    return ["yyyy-MM-dd HH:mm", "yyyy-MM-dd'T'HH:mm:ss.SSSX", "yyyy-MM-dd'T'HH:mm:ssX", "yyyy-MM-dd'T'HH:mmX", "yyyy-MM-dd", "dd/MMM/yy HH:mm", "dd/MMM/yy"];
  };

  const getUserTimeZoneId = () => {
    const user = jiraUtils.getObject(context, 'isCloud')
      ? getJson('/myself', null, true)
      : getJson('/user?username=' + encodeURIComponent(client.$private.sessionJson.name), null, true);
    return user && user.timeZone;
  };

  const requestUser = (locator, failureHandler) => {
    const isCloud = jiraUtils.getObject(context, 'isCloud');
    const path = isCloud ? ('/user?accountId=' + locator) : ('/user?username=' + encodeURIComponent(locator));
    return getJson(path, failureHandler, true);
  };

  const getAttachmentContent = (project, issue, attachment) => {
    const result = plainConnection.getSync('/attachment/' + attachment.id + '/');
    if (result.isSuccess) {
      return {data: result.responseAsStream};
    } else {
      attachmentContentFailureHandler(result);
      return {data: result.responseAsStream}; // the exception has been written by failure handler
    }
  };

  const getServerInfo = () => {
    const info = getJson('/serverInfo', null, true);
    return {version: info.version, time: info.serverTime, isCloud: info.deploymentType === "Cloud"};
  };

  Object.assign(client, {
    // api obligatory
    getServerInfo: getServerInfo.bind(this), // api
    getProjects: getProjects.bind(this), // api
    getAttachmentContent: getAttachmentContent.bind(this), // api
    getUsers: getUsers.bind(this), // api
    // api optional
    prepareToImport: prepareToImport.bind(this), // api
    getIssues: getIssues.bind(this), // api
    getLinkTypes: getLinkTypes.bind(this), // api
    getProject: getProject.bind(this), // api
    getIssueUpdates: getIssueUpdates.bind(this), // api
    getTimestampFormats: getTimestampFormats.bind(this), // api
    getUserTimeZoneId: getUserTimeZoneId.bind(this), // api
    // internal
    getSkippingFailureHandler: getSkippingFailureHandler.bind(this), // internal
    getProjectPermissions: getProjectPermissions.bind(this), // internal
    getStatuses: getStatuses.bind(this), // internal
    getRole: getRole.bind(this), // internal
    getComponent: getComponent.bind(this),  // internal
    getIssue: getIssue.bind(this), // internal
    getIssueChangelog: getIssueChangelog.bind(this), // internal
    getProjectFields: getProjectFields.bind(this), // internal
    prepareFieldSchema: prepareFieldSchema.bind(this), // internal
    requestUser: requestUser.bind(this), // internal
    requestGroup: requestGroup.bind(this), // internal

    security: Security(client, context),
    $private: {
      getJson: getJson.bind(this),
      networkFailureHandler: networkFailureHandler.bind(this),

      issueHandler: IssueHandler(client, context),
      events: Events(client, context),
      fields: Fields(client, context)
    }
  });

  return client;
};

exports.Client = Jira;
