/*
 Copyright 2017 JetBrains s.r.o.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

const cache = require('@jetbrains/youtrack-scripting-api/cache');
const dateTime = require('@jetbrains/youtrack-scripting-api/date-time');
const jiraUtils = require('./utils');
const jiraLinks = require('./links');
const usersDirectory = require('./usersDirectory');

const getName = obj => typeof obj === 'string' ? obj : obj.name;

const trimText = (original, defaultText) => {
  const trimmed = original && original.trim();
  return (trimmed && trimmed.length) ? trimmed : (defaultText || '<empty comment>');
}

function Fields(client, context) {

  const getUsersCache = function () {
    return jiraUtils.getOrCreateObject(context, 'usersCache', () => {
      return cache.create(100, 'users');
    });
  };

  const stateColors = {
    'undefined': 1,
    'new': 15,
    'done': 3,
    'indeterminate': 13
  };

  const Resolution = jiraResolution => {
    return {
      name: getName(jiraResolution),
      colorIndex: stateColors['undefined'],
      description: jiraResolution.description,
      type: 'enum'
    };
  };

  const state = jiraStatus => {
    const res = {name: getName(jiraStatus)};
    if (!jiraStatus.self) {
      const schema = context.getObject('fieldSchema');
      const statuses = schema['statuses'] || (schema['statuses'] = client.getStatuses());
      if (statuses) {
        jiraStatus = statuses.find((status) => {
          return status.name === res.name;
        }) || res.name;
      }
    }
    const statusKey = jiraStatus.statusCategory && jiraStatus.statusCategory.key;
    if (statusKey) {
      res.colorIndex = stateColors[statusKey];
      res.isResolved = statusKey === 'done';
    } else {
      res.colorIndex = stateColors['undefined'];
      res.isResolved = false;
    }

    res.type = 'state';
    console.trace('Jira status converted. Name ' + res.name + ', isResolved: ' + res.isResolved + ', colorIndex: ' + res.colorIndex);
    return res;
  };

  const version = jiraVersion => {
    return {
      name: getName(jiraVersion),
      archived: jiraVersion.archived,
      releaseDate : jiraVersion.releaseDate,
      type: 'version'
    };
  };

  const tag = jiraLabel => {
    return {
      name: trimText(jiraLabel, 'no name tag'),
      type: 'tag'
    }
  };

  const enumField = jiraEnumElement => {
    const name = typeof jiraEnumElement === 'string' ? // as we treat multi-valued fields as enums
      jiraEnumElement : (jiraEnumElement.name || jiraEnumElement.value)
    return {
      name: name,
      description: jiraEnumElement.description,
      type: 'enum'
    };
  };

  const owned = jiraComponent => {
    jiraComponent = client.getComponent(jiraComponent.id);
    const owner = jiraComponent.realAssignee || jiraComponent.assignee;
    return {
      name: getName(jiraComponent),
      owner: owner ? user(owner) : null,
      type: 'ownedField'
    };
  };

  const timeTracking = (foreignFieldValue) => {
    const estimateString = foreignFieldValue['originalEstimate'];
    if (!estimateString) {
      return null;
    }
    return '' + foreignFieldValue['originalEstimateSeconds'] * 1000;
  };

  const deletedUser = {
    id: 'deleted_user',
    name: 'deleted_user',
    login: 'deleted_user',
    accountId: 'deleted_user',
    active: 'false',
    email: null,
    deleted: true,
    banned: true,
    self: 'deleted_user',
    type: 'user'
  };

  const group = original => {
    return {
      'id': original.id || original.name || original.value,
      'name': original.name || original.value,
      'type': 'group'
    };
  };

  const user = original => {
    if (!original) {
      return deletedUser;
    }
    if (original.hasOwnProperty('deleted')) { // this means it's already been filled
      return original;
    }

    const isCloud = jiraUtils.getObject(context, 'isCloud');
    const identifier = isCloud ? original.accountId : (original.name || original.key || original.id);

    if (!identifier) {
      console.warn('Could not find identifier of the Jira ' + (isCloud ? 'cloud' : '') + ' user ' + JSON.stringify(original));
      return deletedUser;
    }

    if (original.accountType === 'app') {
      return {
        id: identifier,
        name: original.displayName,
        login: original.displayName,
        accountId: identifier,
        active: 'false',
        email: null,
        deleted: true,
        banned: true,
        self: original.self,
        type: 'user',
        accountType: 'app'
      };
    }
    usersDirectory.fillFromDirectory(context, original);

    const trim = (str) => {
      if (str.indexOf('"') === 0) {
        str = str.substring(1);
      }
      if (str.indexOf('"') === str.length - 1) {
        str = str.substring(0, str.length - 1);
      }
      return str.trim();
    };

    return getUsersCache().get(identifier, () => {
      let deleted = false;
      if (!original.self) {
        original = client.requestUser(identifier, failure => {
          if (failure.code === 404) { // jira data can contain refs to a deleted user
            deleted = true;
          } else {
            client.$private.networkFailureHandler(failure);
          }
        }) || original;
        usersDirectory.fillFromDirectory(context, original);
      }
      let name = original.name;
      if (!name) {
        console.error("Could not find name of the user:");
        console.error(original);
        name = identifier;
      } else {
        name = trim(name);
      }
      return {
        id: identifier,
        login: name,
        name: name,
        value: name,
        email: original.emailAddress,
        fullName: original.displayName && trim(original.displayName),
        banned: !deleted && !original.active,
        type: 'user',
        deleted: deleted
      };
    });
  };

  const comment = (original, jiraIssue) => {
    const convertedVisibility = client.security.toVisibility(original.visibility, jiraIssue.fields.project);
    return {
      id: original.id,
      author: user(original.author),
      updatedBy: user(original.updateAuthor),
      text: trimText(original.body),
      created: original.created,
      updated: original.updated,
      usesMarkdown: false, // it's still in wiki even for newly created cloud instances
      visibleToUsers: convertedVisibility.users,
      visibleToGroups: convertedVisibility.groups
    };
  };

  const attachment = original => {
    return {
      id: original.id,
      filename: original.filename,
      author: user(original.author),
      updatedBy: user(original.updateAuthor),
      created: original.created,
      updated: original.updated,
      mimeType: original.mimeType,
      size: original.size
    };
  };

  const workItem = original => {
    return {
      id: original.id,
      text: trimText(original.comment),
      author: user(original.author),
      updatedBy: user(original.updateAuthor),
      created: original.created,
      updated: original.updated,
      duration: original.timeSpentSeconds && Math.floor(original.timeSpentSeconds  / 60) || 1
    };
  };

  const dummy = original => {
    return original;
  };

  const period = original => {
    return original && {
      type: 'period',
      value: dateTime.toPeriod(parseInt(original) * 1000).toString()
    };
  };

  const ret = {
    'tags': tag,
    'Resolution': Resolution,
    'author': user,
    'user': user,
    'group': group,
    'enum': enumField,
    'state': state,
    'version': version,
    'ownedField': owned,
    'datetime': dummy,
    'date': dummy,
    'string': dummy,
    'period': period,

    'comments': comment,
    'attachments': attachment,
    'workItems': workItem,

    'links': jiraLinks.issueLink,
    'parent': jiraLinks.parent,
    'com.pyxis.greenhopper.jira:gh-epic-link': jiraLinks.epic
  };
  const multi = singleConverter => (jiraValue, jiraIssue, client) => {
    if (!jiraValue) {
      return null;
    }
    if (!Array.isArray(jiraValue)) { // it's not an array when called to transform events
      jiraValue = [jiraValue];
    }
    return jiraValue.filter(it => it).map(it => singleConverter(it, jiraIssue, client));
  };
  Object.keys(ret).forEach((key) => ret[key] = multi(ret[key]));
  ret.author = user;
  return ret;
}

module.exports = Fields;
