/*
 Copyright 2017 JetBrains s.r.o.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */
const jiraUtils = require('./utils');

const IssueHandler = function (client, context) {
  const getSchema = () => {
    return jiraUtils.getObject(context, "fieldSchema")
  };

  const specialFields = {
    'summary': {
      'name': 'summary',
      'type': 'string',
      'multiValue': false,
      'originalName': 'summary' // different depending on localization
    },
    'description': {
      'name': 'description',
      'type': 'string',
      'multiValue': false,
      'originalName': 'description' // different depending on localization
    },
    'created': {
      'name': 'created',
      'type': 'datetime',
      'multiValue': false,
      'originalName': 'created' // different depending on localization
    },
    'updated': {
      'name': 'updated',
      'type': 'datetime',
      'multiValue': false,
      'originalName': 'updated' // different depending on localization
    },
    'issuetype': {
      'name': 'Type',
      'type': 'enum',
      'multiValue': false,
      'originalName': 'Issue Type'
    },
    'priority': {
      'name': 'Priority',
      'type': 'enum',
      'multiValue': false,
      'originalName': 'Priority'
    },
    'duedate': {
      'name': 'Due date',
      'type': 'date',
      'multiValue': false,
      'originalName': 'Due Date'
    },
    'assignee': {
      'name': 'Assignee',
      'type': 'user',
      'multiValue': false,
      'originalName': 'Assignee'
    },
    'versions': {
      'name': 'Affected versions',
      'type': 'version',
      'multiValue': true,
      'originalName': 'Affects Version/s'
    },
    'components': {
      'name': 'Subsystems',
      'type': 'ownedField',
      'multiValue': true,
      'originalName': 'Component/s'
    },
    'fixVersions': {
      'name': 'Fix versions',
      'type': 'version',
      'multiValue': true,
      'originalName': 'Fix Version/s'
    },
    'status': {
      'name': 'State',
      'type': 'state',
      'multiValue': false,
      'originalName': 'Status'
    },
    'resolution': {
      'name': 'Resolution',
      'type': 'enum',
      'multiValue': false,
      'originalName': 'Resolution'
    },
    'timeoriginalestimate': {
      'name': 'Estimation',
      'type': 'period',
      'multiValue': false,
      'originalName': 'timeoriginalestimate'
    },
    'timespent': {
      'name': 'Spent Time',
      'type': 'period',
      'multiValue': false,
      'originalName': 'timespent'
    },
    'watchers': {
      'name': 'watchers',
      'type': 'user',
      'multiValue': true,
      'originalName': 'watches'
    },
    'voters': {
      'name': 'voters',
      'type': 'user',
      'multiValue': true,
      'originalName': 'votes'
    },
    'author': {
      'name': 'author',
      'type': 'user',
      'multiValue': false,
      'originalName': 'reporter'
    }
  };
  Object.keys(specialFields).forEach((id) => specialFields[id].id = id);

  const fieldIdsMapping = {
    'summary': 'summary',
    'description': 'description',
    'created': 'created',
    'updated': 'updated',
    'reporter': 'author',
    'labels': 'tags',
    'attachment': 'attachments',
    'issuelinks': 'links'
  };

  const unsupportedFieldTypes = [
    'com.pyxis.greenhopper.jira:gh-sprint' // todo need to handle
  ];

  const projectFieldsToIgnore = [
    'summary',
    'description',
    'created',
    'updated',
    'parent',
    'votes',
    'project',
    'watches',
    'issuelinks',
    'worklog',
    'labels',
    'attachment',
    'comment',
    'author',
    'reporter',
    'issuekey',
    'security'
  ];

  const projectFieldTypesToIgnore = [
    'com.pyxis.greenhopper.jira:gh-epic-link',
    'com.pyxis.greenhopper.jira:gh-epic-label'
  ];

  const issueFieldsToIgnore = [
    'project', // don't need to pass it
    'thumbnail', // YouTrack automatically creates thumbnails for all image attachments
    'timetracking',
    // 'timeoriginalestimate', // value equals to timetracking.timeOriginalEstimateSeconds
    'timeestimate', // value equals to remainingEstimateSeconds
    // 'timespent', // value equals to timetracking.timeSpentSeconds. Further, YouTrack calculates this value
    'aggregatetimeoriginalestimate', // this includes an estimate over all subtasks
    'aggregatetimeestimate', // this includes an estimate over all subtasks
    'aggregatetimespent', // YouTrack calculates this value
    'aggregateprogress', // this includes a progress over all subtasks
    'progress', // quite the same as time spent, or what?
    'workratio', // (timeSpent / originalEstimate) * 100
    'creator', 'lastViewed', // no such things in YouTrack
    'resolutiondate', // handled in client.$private.events.js
    'subtasks' // handled when processing 'parent' field on a subtask side
  ];

  const renameProperties = function (obj, mapping) {
    const newObj = {};
    Object.keys(obj).forEach(function (oldName) {
      newObj[mapping[oldName] || oldName] = obj[oldName];
    });
    return newObj;
  };

  const fillIssue = (jiraIssue, failureHandler, compact) => {
    jiraIssue.fields = renameProperties(jiraIssue.fields, fieldIdsMapping);

    const issueKey = jiraIssue.key;

    if (compact) {
      return jiraIssue;
    }

    const project = jiraIssue.fields.project;
    const projectKey = project.key;

    issueFieldsToIgnore.forEach(function (fieldName) {
      delete jiraIssue.fields[fieldName];  // don't need them
    });
    jiraIssue.fields.project = project; // will need it

    const hasAnyPermission = permissions => {
      const projectSchema = getSchema().projects[projectKey];
      return permissions.filter((permission) => projectSchema.$permissions[permission]).length > 0;
    };

    if (jiraIssue.fields.watches) { // does not exist in cached json as it gets removed inside this if
      jiraIssue.fields.watchers = [];
      if (jiraIssue.fields.watches.watchCount > 0 && hasAnyPermission(['PROJECT_ADMIN', 'VIEW_VOTERS_AND_WATCHERS', 'MANAGE_WATCHER_LIST'])) {
        jiraIssue.fields.watchers = client.$private.getJson('/issue/' + issueKey + '/watchers', failureHandler)['watchers'];
      }
      delete jiraIssue.fields.watches;
    }

    if (jiraIssue.fields.votes) { // does not exist in cached json as it gets removed inside this if
      jiraIssue.fields.voters = [];
      if (jiraIssue.fields.votes.votes > 0 && hasAnyPermission(['PROJECT_ADMIN', 'VIEW_VOTERS_AND_WATCHERS'])) {
        jiraIssue.fields.voters = client.$private.getJson('/issue/' + issueKey + '/votes', failureHandler)['voters'];
      }
      delete jiraIssue.fields.votes;
    }

    const retrieveCollection = (fieldName, collectionName) => {
      const field = jiraIssue.fields[fieldName];
      const ret = (field[collectionName].length === field.total) ? field[collectionName] :
        client.$private.getJson('/issue/' + issueKey + '/' + fieldName, failureHandler)[collectionName];
      delete jiraIssue.fields[fieldName];
      return ret;
    };

    if (jiraIssue.fields.comment) { // does not exist in cached json as it gets removed inside this if
      jiraIssue.fields.comments = retrieveCollection('comment', 'comments');
    }

    if (jiraIssue.fields.worklog) { // does not exist in cached json as it gets removed inside this if
      jiraIssue.fields.workItems = retrieveCollection('worklog', 'worklogs');
      delete jiraIssue.fields.worklog;
    }

    jiraIssue.$$originalFields = jiraIssue.$$originalFields || {};
    Object.keys(jiraIssue.fields).forEach(function (fieldId) {
      if (issueFieldsToIgnore.indexOf(fieldId) > -1) {
        console.trace(issueKey + '. Ignoring field ' + fieldId);
        return;
      }
      console.trace(issueKey + '. Processing field ' + fieldId);
      const prototype = getSchema().findField(fieldId, fieldId);
      const fieldType = prototype && prototype.type;
      if (unsupportedFieldTypes.indexOf(fieldType) > -1) {
        console.trace(issueKey + '. Ignoring field ' + fieldId + ' as it is of an unsupported type ' + fieldType);
        return;
      }
      const fieldInfo = ((prototype && prototype.name) || '') + '(' + fieldType + ', ' + fieldId + ')';
      const converter = client.$private.fields[fieldId] || fieldType && client.$private.fields[fieldType];
      const originalFieldValue = jiraIssue.fields[fieldId];
      delete jiraIssue.fields[fieldId];
      if (jiraIssue.$$originalFields[fieldId]) {
        console.trace(issueKey + '. Field ' + fieldInfo + ' was converted earlier' )
      } else if (converter) {
        console.debug(issueKey + '. Converting Jira field ' + fieldInfo);
        jiraIssue.$$originalFields[fieldId] = originalFieldValue;
        const newId = fieldIdsMapping[fieldId] || (prototype && prototype.name) || fieldId;
        const converted = converter(originalFieldValue, jiraIssue, client);
        if (converted !== undefined) {
          jiraIssue.fields[newId] = converted;
        }
      } else {
        console.warn(issueKey + '. No converter for field ' + fieldInfo + '. Will not import its value.');
      }
    });
    if (jiraIssue.fields.parent) {
      jiraIssue.fields.links.push(jiraIssue.fields.parent[0]);
    }

    jiraIssue.history = [];
    if (!compact) { // place event transformation after issue data fill to reuse data from user cache
      client.$private.events.transformEvents(jiraIssue, failureHandler);
    }

    delete jiraIssue.fields.parent;

    delete jiraIssue.changelog; // everything is to be moved to jiraIssue.history

    delete jiraIssue.fields.project; // it was explicitly added earlier

    jiraIssue.usesMarkdown = false; // it's still in wiki even for newly created cloud instances

    return jiraIssue;
  };

  const defineFieldType = field => {
    const schema = field.schema;
    const custom = schema.custom;
    let type;

    const typesMapping = {
      'number': 'integer',
      'option': 'enum',
      'issuetype': 'enum'
    };

    const elementType = schema.items || schema.type;
    let isMulti = schema.type === 'array';
    if (custom === 'com.pyxis.greenhopper.jira:gh-epic-link') {
      type = custom;
    } else if (schema.custom === 'com.atlassian.jira.plugin.system.customfieldtypes:select' ||
      schema.custom === 'com.atlassian.jira.plugin.system.customfieldtypes:radiobuttons') {
      type = 'enum';
      isMulti = false;
    } else if (isMulti && elementType === 'string') {
      type = 'enum';
    } else {
      type = (typesMapping[elementType] || elementType);
    }
    console.debug('Field "' + field.name + '" will have type ' + type);
    field.type = type;
    field.multiValue = isMulti;
  };

  return {
    defineFieldType: defineFieldType,
    specialFields: specialFields,
    projectFieldsToIgnore: projectFieldsToIgnore,
    projectFieldTypesToIgnore: projectFieldTypesToIgnore,
    unsupportedFieldTypes: unsupportedFieldTypes,
    issueFieldsToIgnore: issueFieldsToIgnore,
    renameProperties: renameProperties,
    fillIssue: fillIssue
  }
};

module.exports = IssueHandler;
