/*
 Copyright 2017 JetBrains s.r.o.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

const defaultLinkTypeMapping = {
  Blocks: {name: 'Depend', inversion: false},
  Relates: {name: 'Relates', inversion: false},
  Related: {name: 'Relates', inversion: false}, // can be found in some jira instances with pretty old data
  Duplicate: {name: 'Duplicate', inversion: true},
  Subtask: {name: 'Subtask', inversion: false}
};

const linkTypes = jiraLinkTypes => {
  const noOtherTypeWithSameName = type => !jiraLinkTypes.some(t => t.name === type.name && t.id !== type.id);

  jiraLinkTypes.forEach(jiraLinkType => {
    console.debug('Processing link type: ' + jiraLinkType.name);
    const capitalizedName = jiraLinkType.name.charAt(0).toUpperCase() + jiraLinkType.name.slice(1);
    const defaultYtLinkTypeStub = defaultLinkTypeMapping[capitalizedName];

    if (defaultYtLinkTypeStub) {
      if (noOtherTypeWithSameName(defaultYtLinkTypeStub)) {
        console.debug('Rename link type: ' + jiraLinkType.name + ' to ' + defaultYtLinkTypeStub.name);
        jiraLinkType.name = defaultYtLinkTypeStub.name;
      }
      jiraLinkType.inversion = defaultYtLinkTypeStub.inversion;
    }

    jiraLinkType.sourceToTarget = jiraLinkType.inversion ? jiraLinkType.inward : jiraLinkType.outward;
    jiraLinkType.targetToSource = jiraLinkType.inversion ? jiraLinkType.outward: jiraLinkType.inward;
  });

  return jiraLinkTypes;
};

const issueLink = jiraIssueLink => {
  const inward = !!jiraIssueLink.inwardIssue;
  return {
    linkName: inward ? jiraIssueLink.type.inward : jiraIssueLink.type.outward,
    target: inward ? jiraIssueLink.inwardIssue : jiraIssueLink.outwardIssue
  };
};

const parent = (parentIssue) => {
  return {
    linkName: 'subtask of',
    target: parentIssue
  };
};

const epic = (jiraEpicKey, jiraIssue, client) => {
  if (jiraIssue.fields.parent) {
    console.warn(jiraIssue.key + ': the issue has both "normal" parent and a link to an epic issue. Skipping the later');
    return undefined;
  }
  const parentIssue = client.getIssue(jiraEpicKey, client.getSkippingFailureHandler([403, 404]), true);
  if (!parentIssue) {
    console.warn(jiraIssue.key + ': could not retrieve epic ' + jiraEpicKey);
    return undefined;
  }
  jiraIssue.fields.parent = [{
    linkName: 'subtask of',
    target: parentIssue
  }];
  return undefined;
};

module.exports = {
  linkTypes: linkTypes,
  issueLink: issueLink,
  parent: parent,
  epic: epic
};