const jiraUtils = require('./utils');
const cache = require('@jetbrains/youtrack-scripting-api/cache');

function Security(client, context) {

  const getRolesCache = function () {
    return jiraUtils.getOrCreateObject(context, 'rolesCache', () => {
      return cache.create(100, 'roles');
    });
  };

  const getRoleUsersAndGroups = function (roleName, project, dontFallBackToAdmins) {
    const role = getRolesCache().get(roleName, () => {
      const projectRoles = client.getRole(project.key, '');
      const roleUrl = projectRoles[roleName];
      if (!roleUrl) {
        console.warn('Could not find role with name ' + roleName + ' in the list of project ' + project.name + ' roles');
        return null;
      }
      const roleId = roleUrl.substr(projectRoles[roleName].lastIndexOf('/') + 1)
      const role = client.getRole(project.key, roleId);
      if (!role) {
        console.warn('Could not get role with id ' + roleId);
      }
      return role;
    });
    if (!role) {
      console.warn('Could not retrieve project ' + project.name + ' Administrator role actors. Will use project lead as the only administrator');
      return {groups: null, users: [project.lead]}
    }
    return getRolesCache().get(role.id + ' ' + project.id, () => {
      console.trace('Retrieving groups and users in role ' + roleName + ' in project ' + project.name)
      const actors = role.actors;
      if (!actors || actors.length === 0) {
        console.warn('Jira role ' + roleName + ' has no actors');
        return {users: null, groups: null};
      }
      const ret = {users: [], groups: []};
      actors.forEach(function (actor) {
        if (actor.type === 'atlassian-group-role-actor') {
          const groupName = actor.name;
          let group = client.requestGroup(groupName)
          if (!group && groupName.indexOf('jira-') === 0) {
            group = client.requestGroup(groupName.substr(5)); // jira-administrators -> administrators
            group = group || client.requestGroup(groupName.charAt(5).toUpperCase() + groupName.substr(6)); // jira-administrators -> Administrators
          }
          if (group) {
            ret.groups.push(group);
          } else {
            console.warn('Could not retrieve groups: ' + actor.name + ', ' + groupName.substr(5) + ', ' +
              (groupName.charAt(5).toUpperCase() + groupName.substr(6)));
          }
        } else if (actor.type === 'atlassian-user-role-actor') {
          ret.users.push(client.$private.fields.user(actor)[0]);
        } else {
          console.warn('Unknown role actor type: ' + actor.type);
        }
      });
      if (ret.groups.length === 0 && ret.users.length === 0) {
        if (dontFallBackToAdmins) {
          console.warn('Could not retrieve project ' + project.name + ' Administrator role actors. Will use project lead as the only administrator');
          return {groups: null, users: [project.lead]}
        } else {
          console.info('No users or groups in role ' + roleName + 'in project ' + project.key + ', using project admins instead');
          return getAdmins(project);
        }
      }
      return ret;
    });
  };

  const getAdmins = project => getRoleUsersAndGroups('Administrators', project, true);

  const getDevelopers = project => getRoleUsersAndGroups( 'Developers', project)

  const toVisibility = function (visibility, project) {
    if (!visibility) {
      return {groups: null, users: null};
    }
    const visibilityInfo = 'visibility settings ' + visibility.value + ' (' + visibility.type + ')';
    console.trace('Fetching ' + visibilityInfo);
    if (!visibility.value) {
      console.warn('Name is empty: ' + visibilityInfo);
      return {groups: null, users: null};
    }
    if (visibility.type === 'group') {
      return {groups: [{name: visibility.value}], users: null};
    } else if (visibility.type === 'role') {
      const roleName = visibility.value;
      if (roleName === 'Developers') {
        return getDevelopers(project);
      }
      if (roleName === 'Administrators') {
        return getAdmins(project);
      }
      return getRoleUsersAndGroups(roleName, project)
    } else {
      console.warn('Unknown visibility type ' + visibility.type);
    }
  };

  return {
    toVisibility: toVisibility,
    getAdmins: getAdmins,
    getDevelopers: getDevelopers
  };
}

module.exports = Security;