const dateTime = require('@jetbrains/youtrack-scripting-api/date-time');
const http = require('@jetbrains/youtrack-scripting-api/http');

MantisIssues = function (context) {

    const params = context.parameters;
    //mantis pages starts from 1 !!!
    const projectIdToSkip = {};

    const mantisPageSize = 50;
    let url = params.loadValue('url');
    const sslKeyName = params.loadValue('sslKeyName');
    const token = params.loadValue('password');
    const connection = new http.Connection(url, sslKeyName)
        .addHeader('Authorization', token)
        .addHeader('Accept', 'application/json');

    const client = {};

    const getTimestampFormats = () => {
        return ["yyyy-MM-dd'T'HH:mm:ssXXX", "EEE, d MMM yyyy HH:mm:ss z", "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'"];
    };

    const headersToMap = (headers) => {
        const result = {};
        for (let key in headers) {
            if (!headers.hasOwnProperty(key)) {
                continue;
            }
            result[headers[key].name.toLowerCase()] = headers[key].value;
        }
        return result;
    };

    const getServerInfo = () => {
        const response = connection.getSync('/api/rest/config');
        const headers = headersToMap(response.headers);
        return {
            version: headers['x-mantis-version'] || 'mock',
            time: headers['last-modified'] || new Date().toISOString()
        };
    };

    const getProjects = () => {
        const mProjects = loadJson('/api/rest/projects');
        return mProjects.projects.map(project => {
            return {
                id: project.id,
                key: project.name,
                name: project.name,
            }
        });
    };


    const getProject = (projectKey) => {
        const project = loadJson('/api/rest/projects/' + projectKey.id).projects[0];
        const cf_size = project.custom_fields.length;
        let value = 10;
        const cfId = function () {
            value++;
            return (cf_size + value).toString()
        };

        return {
            id: project.id,
            name: project.name,
            key: project.name,
            description: project.description,
            lead: {
                login: project.access_level.name,
                name: project.access_level.label
            },
            fields: [
                {
                    id: cfId(),
                    name: 'STR',
                    type: 'text'
                },
                {
                    id: cfId(),
                    name: 'platform',
                    type: 'string'
                },
                {
                    id: cfId(),
                    name: 'os',
                    type: 'string'
                },
                {
                    id: cfId(),
                    name: 'os_build',
                    type: 'string'
                },
                {
                    id: cfId(),
                    name: 'Additional_Information',
                    type: 'text'
                },
                {
                    id: cfId(),
                    name: 'Priority',
                    type: 'enum',
                    values: [
                        {
                            name: 'none'
                        },
                        {
                            name: 'low'
                        },
                        {
                            name: 'normal'
                        },
                        {
                            name: 'high'
                        },
                        {
                            name: 'urgent'
                        },
                        {
                            name: 'immediate'
                        }
                    ]

                },
                {
                    id: cfId(),
                    name: 'Status',
                    multiValue: false,
                    type: 'enum',
                    values: [
                        {
                            name: 'new'
                        },
                        {
                            name: 'feedback'
                        },
                        {
                            name: 'acknowledged'
                        },
                        {
                            name: 'confirmed'
                        },
                        {
                            name: 'assigned'
                        },
                        {
                            name: 'resolved'
                        },
                        {
                            name: 'closed'
                        }
                    ]
                },
                {
                    id: cfId(),
                    name: 'Resolution',
                    multiValue: false,
                    type: 'state',
                    values: [
                        {
                            name: 'open',
                            isResolved: false
                        },
                        {
                            name: 'fixed',
                            isResolved: true
                        },
                        {
                            name: 'reopened',
                            isResolved: false
                        },
                        {
                            name: 'unable to reproduce',
                            isResolved: false
                        },
                        {
                            name: 'not fixable',
                            isResolved: true
                        },
                        {
                            name: 'duplicate',
                            isResolved: false
                        },
                        {
                            name: 'no change required',
                            isResolved: true
                        },
                        {
                            name: 'suspended',
                            isResolved: true
                        },
                        {
                            name: 'won\'t fix',
                            isResolved: true
                        },
                    ]
                },
                {
                    id: cfId(),
                    name: 'Severity',
                    multiValue: false,
                    type: 'enum',
                    values: [
                        {
                            name: 'feature'
                        },
                        {
                            name: 'trivial'
                        },
                        {
                            name: 'text'
                        },
                        {
                            name: 'tweak'
                        },
                        {
                            name: 'minor'
                        },
                        {
                            name: 'major'
                        },
                        {
                            name: 'crash'
                        },
                        {
                            name: 'block'
                        },
                    ]
                },
                {
                    id: cfId(),
                    name: 'Reproducibility',
                    multiValue: false,
                    type: 'enum',
                    values: [
                        {
                            name: 'always'
                        },
                        {
                            name: 'sometimes'
                        },
                        {
                            name: 'random'
                        },
                        {
                            name: 'have not tried'
                        },
                        {
                            name: 'unable to reproduce'
                        },
                        {
                            name: 'N/A'
                        },
                    ]
                },
            ].concat(
                project.custom_fields && project.custom_fields.map(field => {
                    return {
                        id: field.id,
                        name: field.name,
                        type: convertType(field.type),
                        values: fieldValues(field.possible_values)
                    }
                }),
            )
        }
    };

    const fieldValues = (values_string) => {
        return values_string.split('|').filter(function (el) {
            return el !== '';
        }).map(value => {
            return {
                name: value
            }
        });
    };

    const convertType = (type) => {
        console.log('from mantis type: ' + type);
        switch (type) {
            case 'string':
            case 'email':
            case'checkbox':
                return 'string';
            case 'numeric':
                return 'integer';
            case 'float':
                return 'float';
            case 'enum':
            case '9':
            case 'multilist':
                return 'enum';
            case 'date':
                return 'date';
            case 'textarea':
                return 'text';
            default:
                return 'string';
        }
    };

    const getIssues = (project, from, top) => {
        let count = 0;
        let mantisIssues = [];
        let skipIssuesPage = projectIdToSkip[project.id];
        let query = {
            'page_size': mantisPageSize + '',
            'project_id': project.id + '',
            'page': skipIssuesPage + ''
        };
        var issues = [];
        do {
            issues = loadJson('/api/rest/issues', query).issues || [];
            issues.forEach(function (issue) {
                mantisIssues.push(issue);
                count++;
            });
            skipIssuesPage = skipIssuesPage + 1;
            query.page = skipIssuesPage + '';
            projectIdToSkip[project.id] = skipIssuesPage;
        } while (issues.length === mantisPageSize && count < top);

        console.log('Loaded ' + count + ' issues ' + 'project ' + project.name);
        return mantisIssues.map(convertMantisIssue);
    };


    const convertMantisIssue = function (issue) {
        const mantisFields = convertMantisCFs(issue);
        //const mantisHistory = convertMantisHistory(issue);
        //console.log('test12345 mantisHistory ' + JSON.stringify(mantisHistory));
        const defaultFields = {
            summary: issue.summary,
            description: issue.description,
            STR: issue.steps_to_reproduce,
            platform: issue.platform,
            os: issue.os,
            os_build: issue.os_build,
            Additional_Information: issue.additional_information,
            created: issue.created_at,
            updated: issue.updated_at,
            Priority: issue.priority.name,
            Status: issue.status.name,
            Resolution: issue.resolution.name,
            Severity: issue.severity.name,
            attachments: issue.attachments && issue.attachments.map(convertMantisAttachments(issue.id)),
            tags: issue.tags && issue.tags.map(tag => {
                return {
                    id: tag.id,
                    name: tag.name,
                    type: 'tag',
                    visibleToGroups: [
                        'All Users'
                    ]
                }
            }),
            watchers: issue.monitors && issue.monitors.map(watcher => {
                return {
                    id: watcher.id,
                    login: watcher.name,
                    name: watcher.real_name ? watcher.real_name : watcher.name,
                    email: watcher.email,
                    type: 'user'
                }
            }),
            visibleToGroups: setVisible(issue.view_state.name, issue.project.name),
            comments: issue.notes && issue.notes.map(convertMantisComment(issue.project.name)),
            author: {
                id: issue.reporter.id,
                login: issue.reporter.name,
                name: issue.reporter.name,
                email: issue.reporter.email,
                type: 'user'
            },
            assignee: issue.handler && {
                id: issue.handler.id,
                login: issue.handler.name,
                name: issue.handler.name,
                email: issue.handler.email,
                type: 'user'
            }
        };

        return {
            id: issue.id,
            key: issue.id + '',
            fields: Object.assign({}, defaultFields, mantisFields)
            //history: mantisHistory
        };
    };

    const convertMantisHistory = function (issue) {
        let id = 0;
        let res = [];
        if (issue.history) {
            issue.history.map(history => {
                if (history.type.name === "field-updated") {
                    id = id + 1;
                    let name = history.field.name;
                    const ret = {
                        id: id + '',
                        timestamp: history.created_at,
                        author: {
                            id: history.user.id,
                            login: history.user.name,
                            name: history.user.name,
                            email: history.user.email,
                            type: 'user'
                        },
                        fieldChanges: {}
                    };
                    ret.fieldChanges[name] = {
                        removedValues: history.old_value.name ? history.old_value.name : history.old_value,
                        addedValues: history.new_value.name ? history.new_value.name : history.new_value
                    };
                    res.push(ret)
                }
            });
        }
        return res;
    };

    const convertMantisAttachments = function (issueID) {
        return (attachment) => {
            return {
                id: attachment.id + '-' + issueID,
                created: attachment.created_at,
                filename: attachment.filename,
                mimeType: attachment.content_type.split(';')[0],
                charset: attachment.content_type.split('charset=')[1],
                author: {
                    id: attachment.reporter.id,
                    login: attachment.reporter.name,
                    name: attachment.reporter.name,
                    email: attachment.reporter.email,
                    type: 'user'
                }
            };
        };
    };

    const getContent = (issueID, attachID) => {
        const mAttachment = loadJson('/api/rest/issues/' + issueID + '/files/' + attachID);
        return mAttachment.files[0].content
    };


    const convertMantisCFs = function (issue) {
        const result = {};
        if (issue.custom_fields !== undefined) {
            issue.custom_fields.forEach(field => {
                if (field.value !== '') {
                    result[field.field.name] = field.value;
                }
            });
        }
        return result;
    };

    const convertMantisComment = function (projectName) {
        return (comment) => {
            return {
                id: comment.id + '',
                text: comment.text,
                visibleToGroups: setVisible(comment.view_state.name, projectName),
                author: {
                    id: comment.reporter.id,
                    login: comment.reporter.name,
                    name: comment.reporter.name,
                    email: comment.reporter.email,
                    type: 'user'
                },
                updated: comment.updated_at,
                created: comment.created_at
            };
        };
    };

    const setVisible = (state, project) => {
        if (state === 'private')
            return [project + ' Team'];
    };

    const loadJson = (uri, queryParams) => {
        const result = connection.getSync(uri, queryParams);
        if (!result.isSuccess) {
            console.log('Failed to load data from URL ' + uri);
            throw 'Failed to load data from URL ' + uri + ' with code ' + result.code;
        } else {
            return JSON.parse(result.response);
        }
    };

    const getUsers = function () {
        return [];
    };

    const getAttachmentContent = function (project, issue, attachment) {
        const attachID = attachment.id.split("-")[0];
        const issueID = attachment.id.split("-")[1];
        return {
            data: getContent(issueID, attachID)
        }
    };

    Object.assign(client, {
        getProjects: getProjects.bind(this), // api
        getProject: getProject.bind(this), // api
        getLinkTypes: () => [],
        getServerInfo: getServerInfo.bind(this), // api
        getAttachmentContent: getAttachmentContent.bind(this),
        getUsers: getUsers.bind(this),
        getIssues: getIssues.bind(this), // api
        getTimestampFormats: getTimestampFormats.bind(this) // api
    });
    return client;

};
exports.Client = MantisIssues;
