// https://www.redmine.org/projects/redmine/wiki/Rest_api

const connection = require('./connection');
const utils = require('./utils');

//TODO Wiki to markdown


RedmineIssues = function (context) {

    const client = {};

    const connect = connection.makeHttpConnection(context);
    const loadJSON = connection.loadJsonFactory(connect);

    const pageSize = 25;
    let articlesCounter = 1;

    let usersData = {};
    let workItemsData = {};
    let customFields = [];
    const groups = {};
    const articlesMap = {};
    //used for history reasons, currently unnecessary
    let projects = [];

    const miscQuery = {
        'include': 'attachments,relations,journals,watchers,children'
    };

    const getTimestampFormats = () => {
        return ["yyyy-MM-dd'T'HH:mm:ss'Z'", "yyyy-MM-dd"];
    };


    const myAccountGetter = () => {
        const account = loadJSON('/users/current.json').user || {};
        return {
            id: account.id,
            login: account.login,
            name: account.firstname,
            fullName: account.firstname + ' ' + account.lastname,
            email: account.mail,
        }
    }
    const myAccount = myAccountGetter();

    const getUsersData = () => {
        const result = {};
        //checking banned users
        const banned = loadJSON('/users.json?', {'status': '3'}).users || [];
        banned && banned.forEach(user => {
            user['banned'] = true;
            result[user.id] = user;
        })
        //?status= parameter is needed for fetching of non-active users(including banned)
        const data = loadJSON('/users.json?', {'status': ''}).users || [];
        data.forEach(user => {
            if (!result.hasOwnProperty(user.id)) result[user.id] = user;

        });
        return result;
    }

    const getWorkItems = () => {
        let result = {};
        const data = loadJSON('time_entries.json').time_entries || {};
        data.forEach(workItem => {
            if (!result.hasOwnProperty(workItem.issue.id)) result[workItem.issue.id] = [];
            result[workItem.issue.id].push(workItem);
        });
        //needed to sort array of object per id, otherwise the elder one will be only used
        Object.keys(result).forEach(el => {
            result[el].sort((a, b) => a.id - b.id)
        });
        return result;
    }


    const prepareToImport = () => {
        console.log('Preparing to leave Redmine... ( ͡❛ ͜ʖ ͡❛)');
        console.log('Fetching usersData...')
        usersData = getUsersData();
        console.log('Getting workItems...')
        workItemsData = getWorkItems();
        getGroups();

    }

    const headersToMap = (headers) => {
        const result = {};
        for (let header in headers) {
            if (!headers.hasOwnProperty(header)) {
                continue
            }
            result[headers[header].name.toLowerCase()] = headers[header].value
        }
        return result;
    };

    const getServerInfo = () => {
        const response = connect.getSync('?');
        const headers = headersToMap(response.headers);
        const server = headers['server'];
        return {
            version: server,
            time: new Date(headers['date']).toISOString()
        }
    };

    const getCustomFields = () => {

        const checkCfType = (field) => {
            //filtering unnecessary custom field types
            const cfArray = ['issue', 'project', 'version', 'issue_priority'];
            return cfArray.indexOf(field.customized_type) > -1;
        }

        const customFields = loadJSON('/custom_fields.json').custom_fields;
        return customFields.filter(checkCfType).map(cf => {
            return {
                id: cf.id,
                name: cf.name,
                multiValue: cf.multiple,
                type: utils.convertType(cf.field_format),
                values: cf.possible_values && cf.possible_values.map(field => {
                    return {name: field.label}
                }) || [],
            }
        });
    }

    const getProjects = () => {
        let result;

        const rProjects = loadJSON('/projects.json');
        result =  rProjects.projects.map(project => {
            return {
                id: project.id,
                key: utils.cleanKeyName(project.identifier),
                name: project.name,
            }
        });

        projects = result;

        return result;

    };

    const getProject = (projectKey) => {
        customFields = getCustomFields();
        const project = loadJSON('/projects/' + projectKey.id + '.json', {'include': 'issue_categories,trackers'}).project;
        //getting subsystems
        const trackers = project.trackers;

        //getting members, adding them to the project's team
        //https://www.redmine.org/projects/redmine/wiki/Rest_Memberships
        const memberships = loadJSON('/projects/' + projectKey.id + '/memberships.json').memberships;
        groups[projectKey.name + ' Team'] = [];
        groups[projectKey.name + ' Team'] = memberships.filter(el => el.user).map(el => {
            return {id: el.user.id, fullName: el.user.name}
        });

        const adminGroups = memberships.filter(el => el.group);
        const subsystems = loadJSON('/projects/' + projectKey.id + '/issue_categories.json').issue_categories || [];
        const statuses = loadJSON('/issue_statuses.json').issue_statuses;
        const priorities = loadJSON('/enumerations/issue_priorities.json').issue_priorities;
        const versions = loadJSON('/projects/' + projectKey.id + '/versions.json').versions || [];

        //filling fieldsMap for history conversion
        utils.toFieldMap([trackers, subsystems, usersData, statuses, priorities, versions, customFields, projects], ['tracker_id', 'category_id', 'users', 'status_id', 'priority_id', 'fixed_version_id', 'cf', 'project_id']);

        //checking if the project has custom fields
        const hasCustomFields = loadJSON('/issues.json', {
            project_id: projectKey.id,
            limit: '1'
        }).issues[0].custom_fields ? customFields.length > 1 : false;
        const defaultFields = {
            id: project.id,
            key: utils.cleanKeyName(project.identifier),
            name: project.name,
            description: project.description,
            lead: {
                login: myAccount.login,
                name: myAccount.login,
                fullName: myAccount.fullName,
                email: myAccount.email,
            },
            fields: [
                {
                    id: 'State',
                    name: 'State',
                    multiValue: false,
                    type: 'state',
                    values: statuses.map(status => {
                        return {
                            name: status.name,
                            isResolved: status.is_closed,
                        }
                    })
                },
                {
                    id: 'Estimation',
                    name: 'Estimation',
                    multiValue: false,
                    type: 'period',
                },
                {
                    id: 'Spent time',
                    name: 'Spent time',
                    multiValue: false,
                    type: 'period',
                },
                {
                    id: 'Priority',
                    name: 'Priority',
                    multiValue: false,
                    type: 'enum',
                    values: priorities.map(priority => {
                        name: priority.name
                    })
                },
                {
                    id: 'Type',
                    name: 'Type',
                    multiValue: false,
                    type: 'enum',
                    values: trackers.map(type => {
                            name: type.name
                        }
                    ),
                },
                {
                    id: 'Fix versions',
                    name: 'Fix versions',
                    multiValue: true,
                    type: 'version',
                },
                {
                    id: 'Due Date',
                    name: 'Due Date',
                    multiValue: false,
                    type: 'date',
                },
                {
                    id: 'Start Date',
                    name: 'Start Date',
                    multiValue: false,
                    type: 'date',
                },
            ],
            adminGroups: adminGroups.map(el => {
                return {id: el.group.id, name: el.group.name}
            }),
        };

        if (subsystems.length) {
            const subField = {
                id: 'Subsystem',
                name: 'Subsystem',
                multiValue: false,
                type: 'ownedField',
                values: subsystems.length && subsystems.map(el => {
                    let owner = usersData[el.assigned_to.id];
                    return {
                        name: el.name,
                        owner: {
                            login: owner.login,
                            name: owner.login,
                            fullName: owner.firstname + ' ' + owner.lastname,
                            email: owner.mail,
                        },
                    }
                })
            };

            defaultFields.fields.push(subField);
        }

        if (hasCustomFields) defaultFields.fields.concat(customFields);

        return defaultFields;

    };

    const getGroups = () => loadJSON('/groups.json').groups.forEach(group => groups[group.name] = group);


    const getGroupId = (name) => {
        return groups.hasOwnProperty(name) ? groups[name].id : '';

    }
    const getUsers = (group, skip, top) => {
        let users;

        //checking if a group is a team
        if (group.name.indexOf('Team') !== -1) {
            users = groups[group.name].map(user => user.id);
        } else {
            const id = getGroupId(group.name);
            const query = {'group_id': id + '', 'limit': top + '', 'offset': skip + ''};
            users = loadJSON('/users.json?', query).users.map(user => user.id);
        }

        return users.map(user => {
            const currentUser = usersData[user];
            return {
                name: currentUser.login,
                fullName: currentUser.firstname + ' ' + currentUser.lastname,
                email: currentUser.mail,
            }
        });
    };


    const getAttachmentContent = (project, issue, attachment) => {
        try {
            const res = loadJSON('/attachments/' + attachment.id + '.json').attachment;
            const mediaLocation = res.content_url.substring(res.content_url.indexOf('attachments'));
            const response = connect.getSync(mediaLocation);
            return {
                data: response.responseAsStream,
                metadata: {
                    mimeType: res.content_type
                }
            }
        } catch (e) {
            console.debug('Failed to download attachment :( ', e.message)
        }
    };

    // requesting additional info as general issues endpoint doesn't provide required info
    const getAdditionalInfo = (issue) => {
        return loadJSON('/issues/' + issue.id + '.json', miscQuery).issue
    }


    const getIssues = (project, after, top) => {
        let redmineIssues = [];
        let count = 0;
        let currentOffset = 0;

        const query = {
            'status_id': '*',
            'limit': pageSize + '',
            'project_id': project.id + '',
            'offset': currentOffset + '',

        };

        let tempIssues = [];

        do {
            tempIssues = loadJSON('/issues.json', query).issues || [];
            tempIssues.forEach(issue => {
                redmineIssues.push(issue);
                count++;
            })
            currentOffset += pageSize;
            query.offset = currentOffset + '';
        } while (tempIssues.length === pageSize && count < top)
        console.log('Loaded ' + count + ' issues from ' + project.name);
        console.log('Getting journals, relations and watchers...');
        redmineIssues = redmineIssues.map(getAdditionalInfo);
        return redmineIssues.map(convertRedmineIssues);

    };

    const redmineAttachmentsConverter = attachment => {
        const author = usersData[attachment.author.id];
        return {
            id: attachment.id + '',
            created: attachment.created_on,
            filename: attachment.filename,
            mimeType: attachment.content_type,
            author: {
                id: author.id,
                name: author.login,
                type: 'user'
            },
            charset: 'UTF-8',
        };
    };

    const convertRedmineComment = projectName => {
        try {
            return (comment) => {
                const author = usersData[comment.user.id];
                return {
                    id: comment.id + '',
                    text: comment.notes.trim(),
                    author: {
                        id: author.id,
                        login: author.login,
                        name: author.login,
                        fullName: author.firstname + ' ' + author.lastname,
                    },
                    attachments: comment.details && comment.details.filter(issue => issue.property === 'attachment').map(comment => {
                        return {
                            id: comment.name,
                            name: comment.new_value,
                            filename: comment.new_value,
                        }
                    }),
                    created: comment.created_on,
                    updated: comment.updated_on,
                    visibleToGroups: utils.setVisible(comment.private_notes, projectName),

                };


            };
        } catch (e) {
            console.log('Converter failed,', e.message);
            console.log('Current comment: ', comment)
            console.debug(e.stack)
        }
    };

    const convertRedmineHistory = issue => event => {
        const currentAuthor = usersData[event.user.id];
        const currResult = {};
        let attrObj;

        event.details.forEach(el => {

                if (el.property === 'relation') {
                    const relationName = utils.convertRelations(el.name);
                    currResult['links'] = {};
                    currResult['links'] = {
                        'removedValues': {
                            'type': 'link',
                            'linkName': relationName,
                            'target': el.old_value ? el.old_value + '' : '',
                        },
                        "addedValues": {
                            'type': 'link',
                            'linkName': relationName,
                            'target': el.new_value ? el.new_value + '' : ''
                        },
                    }

                } else if (el.property === 'attr' && el.name === 'parent_id') {
                    currResult['links'] = {};
                    currResult['links'] = {
                        'removedValues': {
                            'type': 'link',
                            'linkName': 'subtask of',
                            'target': el.old_value ? el.old_value + '' : '',
                        },
                        'addedValues': {
                            'type': 'link',
                            'linkName': 'subtask of',
                            'target': el.new_value ? el.new_value + '' : ''
                        },
                    };

                } else if (el.property === 'attr' && el.name === 'status_id' || el.name === 'fixed_version_id' || el.name === 'tracker_id' || el.name === 'category_id' || el.name === 'assigned_to_id' || el.name === 'is_private' || el.name === 'priority_id') {
                    attrObj = utils.convertHistoryField(el, issue);
                    console.log('ATTROBJ_STATE', JSON.stringify(attrObj));

                    if (!attrObj.old && !attrObj.new) return;

                    currResult[attrObj.name] = {};
                    currResult[attrObj.name] = {
                        removedValues: attrObj.old ? [{type: attrObj.type, name: attrObj.old}] : [],
                        addedValues: attrObj.new ? [{type: attrObj.type, name: attrObj.new}] : []
                    };

                } else if (el.property === 'attr') {
                    attrObj = utils.convertHistoryField(el, issue);
                    console.log('ATTROBJ', JSON.stringify(attrObj));

                    if(!attrObj) return;

                    currResult[attrObj.name] = {};
                    currResult[attrObj.name] = {
                        removedValues: attrObj.old ? [{type: attrObj.type, value: attrObj.old}] : [],
                        addedValues: attrObj.new ? [{type: attrObj.type, value: attrObj.new}] : []
                    };

                } else if (el.property === 'cf') {
                    attrObj = utils.convertHistoryCfField(el);
                    console.log('ATTROBJ_CF', JSON.stringify(attrObj));

                    currResult[attrObj.name] = {};
                    currResult[attrObj.name] = {
                        removedValues: attrObj.old ? [{type: attrObj.type, value: attrObj.old, name: attrObj.old}] : [],
                        addedValues: attrObj.new ? [{type: attrObj.type, value: attrObj.new, name: attrObj.new}] : []
                    };

                }
                //there is no need for attachment history atm
                /*else if (el.property === 'attachment') {
                currResult['Attachment'] = {};
                currResult['Attachment'] = {
                    'removedValues': {
                        'type': 'string',
                        'value': el.old_value ? el.old_value + '' : '',
                    },
                    'addedValues': {
                        'type': 'string',
                        'value': el.new_value ? el.new_value + '' : '',
                    },
                };
            }*/
            }
        )

        const result = {
            id: event.id + '',
            author: {
                id: currentAuthor.id + '',
                login: currentAuthor.login,
                name: currentAuthor.login,
                fullName: currentAuthor.firstname + ' ' + currentAuthor.lastname,

            },
            timestamp: event.created_on,
        }
        result.fieldChanges = {};

        console.log('currentResult', currResult);

        Object.keys(currResult).forEach(key => {
            result.fieldChanges[key] = {};
            result.fieldChanges[key] = currResult[key]
        })

        console.log('RESULT', JSON.stringify(result));


        return result;

    }

    const getCustomFieldsFromIssue = (issue) => {
        const result = {};
        const cfMap = {};

        //creating customFields map for setting the custom field's type
        customFields.forEach(cf => {
            cfMap[cf.id] = {};
            cfMap[cf.id] = cf
        })

        if (issue.custom_fields && issue.custom_fields.length) {
            issue.custom_fields.forEach(field => {
                result[field.name] = {
                    type: cfMap[field.id].type,
                    value: field.value,
                    name: field.name, //some of the fields require 'name' instead of 'value'
                };
            });
        }
        return result;

    }

    const convertRedmineIssues = (issue) => {
        //getting additional info for proper user creation
        const author = usersData[issue.author.id];
        let assignee = {};
        const children = issue.children && issue.children.map(child => {
            return {
                linkName: 'parent for',
                target: child.id + '',
            }
        }) || [];

        if (issue.assigned_to) {
            assignee = issue.assigned_to.id === issue.author.id ? author : usersData[issue.assigned_to.id]
        }
        const defaultFields = {
            State: {
                type: 'state',
                name: issue.status.name,
            },
            summary: {
                type: 'string',
                value: issue.subject,
            },
            description: {
                type: 'text',
                value: issue.description,
            },
            Type: {
                type: 'enum',
                name: issue.tracker.name,
            },
            created: issue.created_on,
            updated: issue.updated_on,
            Priority: {
                type: 'enum',
                name: issue.priority.name,
            },
            'Start Date': issue.start_date && {
                type: 'date',
                value: issue.start_date
            },
            'Due Date': issue.due_date && {
                type: 'date',
                value: issue.due_date
            },
            Estimation: issue.estimated_hours && {
                type: 'period',
                value: utils.floatConverter(issue.estimated_hours),
            },
            'Fix versions': issue.fixed_version && {
                type: 'version',
                name: issue.fixed_version.name
            },
            Subsystem: issue.category && {
                type: 'ownedField',
                name: issue.category.name,
            },
            author: {
                id: author.id,
                login: author.login,
                email: author.mail,
                name: author.login,
                fullName: author.firstname + ' ' + author.lastname,
                banned: author.banned || false,
                type: 'user'
            },
            assignee: issue.assigned_to && {
                id: assignee.id,
                login: assignee.login,
                email: assignee.mail,
                name: assignee.login,
                banned: assignee.banned || false,
                fullName: assignee.firstname + ' ' + assignee.lastname,
                type: 'user'
            },
            links: issue.relations && issue.relations.map(rel => {
                return {
                    linkName: utils.convertRelations(rel.relation_type),
                    target: rel.issue_to_id + '',
                }
            }).concat(children),
            watchers: issue.watchers && issue.watchers.map(watcher => {
                const currentWatcher = usersData[watcher.id];
                return {
                    id: currentWatcher.id,
                    name: currentWatcher.login,
                    login: currentWatcher.login,
                    email: currentWatcher.mail,
                    fullName: currentWatcher.firsname + ' ' + currentWatcher.lastname,
                    banned: currentWatcher.banned || false,
                    type: 'user'
                }
            }),
            workItems: workItemsData[issue.id] && workItemsData[issue.id].map(item => {
                const author = usersData[item.user.id];
                return {
                    id: item.id + '',
                    text: item.comments.trim(),
                    author: author.login,
                    created: item.created_on,
                    updated: item.updated_on,
                    duration: item.hours * 60,

                }
            }),
            visibleToGroups: utils.setVisible(issue.is_private, issue.project.name),
            attachments: issue.attachments && issue.attachments.map(redmineAttachmentsConverter),
            comments: issue.journals && issue.journals.filter(comment => comment.notes).map(convertRedmineComment(issue.project.name)),
            resolved: issue.closed_on,
        };
        const redmineFields = getCustomFieldsFromIssue(issue);
        return {
            id: issue.id,
            key: issue.id + '', //faking until we make em'
            fields: Object.assign({}, defaultFields, redmineFields),
            history: issue.journals && issue.journals.filter(el => el.details.length).map(convertRedmineHistory(issue)).filter(el => el !== null).sort((a, b) => a.timestamp - b.timestamp),
        };
    }

    const getArticles = (projectInfo, after, top) => {
        const query = {'offset': after + '', 'limit': top + ''};
        const articles = loadJSON('/projects/' + projectInfo.id + '/wiki/index.json', query).wiki_pages;
        return articles.map(article => {

            const defaultFields = {
                id: articlesCounter + '',
                key: articlesCounter++ + '',
                aId: article.title,
            };


            const articleDetails = loadJSON('projects/' + projectInfo.id + '/wiki/' + defaultFields.aId + '.json', {'include': 'attachments'}).wiki_page;

            const author = usersData[articleDetails.author.id];

            const articleFields = {
                fields: {
                    summary: articleDetails.title.replace(/_/g, ' '),
                    content: articleDetails.text,
                    author: {
                        id: author.id,
                        login: author.login,
                        email: author.mail,
                        name: author.login,
                        fullName: author.firstname + ' ' + author.lastname,
                        banned: author.banned || false,
                        type: 'user'
                    },
                    parent: articleDetails.parent && articlesMap[projectInfo.id][articleDetails.parent.title].id,
                    attachments: articleDetails.attachments && articleDetails.attachments.map(redmineAttachmentsConverter),
                    created: articleDetails.created_on,
                    updated: articleDetails.updated_on,
                    comments: articleDetails.comments && [{
                        text: articleDetails.comments.trim(),
                        author: {
                            id: author.id,
                            login: author.login,
                            email: author.mail,
                            name: author.login,
                            fullName: author.firstname + ' ' + author.lastname,
                            banned: author.banned || false,
                            type: 'user'
                        },
                    }] || [],
                }
            }
            return Object.assign({}, defaultFields, articleFields);
        });


    }

    Object.assign(client, {
        getProjects: getProjects.bind(this), // api
        getProject: getProject.bind(this), // api
        getLinkTypes: () => [{
            id: '1',
            name: 'Copied',
            sourceToTarget: 'copied to',
            targetToSource: 'copied from',
        }, {
            id: '2',
            name: 'Blocks',
            sourceToTarget: 'blocks',
            targetToSource: 'blocked by',
        },
            {
                id: '3',
                name: 'Precedes',
                sourceToTarget: 'follows',
                targetToSource: 'precedes',
            },],
        getServerInfo: getServerInfo.bind(this), // api
        getAttachmentContent: getAttachmentContent.bind(this),
        getUsers: getUsers.bind(this),
        // getUsers: () => [],
        prepareToImport: prepareToImport.bind(this),
        getIssues: getIssues.bind(this), // api
        getTimestampFormats: getTimestampFormats.bind(this), // api
        getArticles: getArticles.bind(this),
    });
    return client;

}
;
exports.Client = RedmineIssues;
