/*
 Copyright 2017 JetBrains s.r.o.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

/**
 * This module contains functionality that lets you search for issues in YouTrack.
 *
 * @module @jetbrains/youtrack-scripting-api/search
 */

var Issue = require('./entities').Issue;

/**
 * Returns issues that match a search query.
 * If a sort order is not specified explicitly in the query, the issues that are returned are sorted in random order.
 *
 * @example
 * var search = require('@jetbrains/youtrack-scripting-api/search');
 * ...
 * var query = 'for: me State: {In Progress} issue id: -' + issue.id;
 * var inProgress = search.search(issue.project, query, ctx.currentUser);
 * if (inProgress.isNotEmpty()) {
 *   // Do smth with a found set of issues.
 * }
 *
 * @param {WatchFolder} [folder] The project, tag, or saved search to set as the search context.
 * If the value for this parameter is not provided, the search includes all issues.
 * This is equivalent to a search that is performed in the user interface with the context set to Everything.
 * @param {string} query A YouTrack search query.
 * @param {User} [user] The user account that executes the search query.
 * The list of issues that is returned excludes issues that the specified user does not have permission to view.
 * If the value for this parameter is not provided, the search query is executed on behalf of the current user.
 */
exports.search = function(folder, query, user) {
  if (typeof folder === 'string') {
    query = folder;
    folder = null;
  }
  return user ? Issue.search(folder, query, user) : Issue.search(folder, query);
};