// Copyright 2018 JetBrains s.r.o.
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//  http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
@file:JsQualifier("\$module_youtrack_entities")
@file:Suppress("unused")

package jetbrains.youtrack.plugin.api

import kotlin.Boolean
import kotlin.Int
import kotlin.Long
import kotlin.String
import kotlin.Unit
import kotlin.collections.Iterable

/**
 * Represents an agile board and the set of sprints that belong to the board.
 */
open external class Agile : BaseEntity {
    /**
     * The name of the agile board.
     */
    val name: String

    /**
     * The set of sprints that are associated with the board.
     */
    val sprints: Iterable<Sprint>?

    /**
     * The user who created the board.
     */
    val author: User?

    /**
     * Finds a specific sprint by name.
     * @param name The name of the sprint.
     * @return If a sprint with the specified name is found, the corresponding Sprint object is returned. Otherwise, the return value is null.
     */
    fun findSprintByName(name: String?): Sprint?

    /**
     * Returns the sprints that an issue is assigned to on an agile board.
     * @param issue The issue for which you want to get the sprints that it is assigned to.
     * @return The sprints that the issue is assigned to on the agile board.
     *
     * @since 2018.1.39547
     */
    fun getIssueSprints(issue: Issue?): Iterable<Sprint>?
    companion object {
        /**
         * Returns a set of agile boards that have the specified name.
         * @param name The name of an agile board.
         * @return A set of agile boards that are assigned the specified name.
         */
        fun findByName(name: String?): Iterable<Agile>?
    }
}

/**
 * The base class for issue comment.
 */
open external class BaseComment : BaseEntity {
    /**
     * The set of attachments that are attached to the comment.
     *
     * @since 2018.1.40030
     */
    val attachments: Iterable<IssueAttachment>?

    /**
     * When `true`, the comment text is parsed as Markdown. When `false`, the comment text is parsed as YouTrack Wiki.
     * Changing this value does not transform the markup from one syntax to another.
     *
     * @since 2017.4.38870
     */
    var isUsingMarkdown: Boolean?
}

/**
 * The base class for issue work items.
 */
open external class BaseWorkItem : BaseEntity {
    /**
     * The user to whom the work is attributed in the work item.
     */
    val author: User

    /**
     * The date when the work item was created.
     */
    val created: Long?

    /**
     * The user who added the work item to the issue.
     */
    val creator: User

    /**
     * The work item description.
     */
    var description: String?

    /**
     * The work item type.
     */
    val type: WorkItemType?

    /**
     * The date when the work item was last updated.
     */
    val updated: Long?

    /**
     * When `true`, the work item description is parsed as Markdown. When `false`, the work item description is parsed as YouTrack Wiki.
     * Changing this value does not transform the markup from one syntax to another.
     *
     * @since 2017.4.38870
     */
    var isUsingMarkdown: Boolean?
}

/**
 * Represents a value that is stored in a custom field that stores a build type.
 */
open external class Build : Field {
    /**
     * The date and time when the build was assembled.
     */
    val assembleDate: Long?
    companion object {
        /**
         * Field type. Used when defining rule requirements.
         */
        val fieldType: String?
    }
}

/**
 * Represents a custom field in a project that stores an predefined set of values.
 */
open external class BundleProjectCustomField : ProjectCustomField {
    /**
     * The set of values that is available for the custom field.
     */
    val values: Iterable<Field>?

    /**
     * Returns a value with the specified name in the set of values for a custom field.
     * @param name The name of the field value to search for.
     * @return The value with the specified name in the set of values for the custom field.
     */
    fun findValueByName(name: String?): Field?

    /**
     * Returns a value that is assigned a specified position in the set of values for a custom field.
     * @param ordinal The position of the field value to search by.
     * @return The value that is assigned the specified position in the set of values for the custom field.
     */
    fun findValueByOrdinal(ordinal: Int?): Field?
}

/**
 * An entity that retrieves VCS changes and creates their representation in YouTrack.
 */
open external class ChangesProcessor : BaseEntity {
    /**
     * The VCS server that the processor connects to.
     *
     * @since 2018.1.38923
     */
    val server: VcsServer
}

/**
 * Represents a value in a custom field that stores a predefined set of values.
 */
open external class EnumField : Field {
    companion object {
        /**
         * Field type. Used when defining rule requirements.
         */
        val fieldType: String?
    }
}

/**
 * Represents a value that is stored in a custom field.
 * The custom fields themselves are represented by the Fields class.
 */
open external class Field : BaseEntity {
    /**
     * The index value of the color that is assigned to the value in the custom field.
     */
    val colorIndex: Int?

    /**
     * The description of the value as visible in the administrative UI for custom fields.
     */
    val description: String?

    /**
     * The name of the value, which is also stored as the value in the custom field.
     */
    val name: String

    /**
     * The position of the value in the set of values for the custom field.
     */
    val ordinal: Int?

    /**
     * The background color of the value in the custom field as it is displayed in YouTrack.
     */
    val backgroundColor: String?

    /**
     * The foreground color of the value in the custom field as it is displayed in YouTrack.
     */
    val foregroundColor: String?

    /**
     * String representation of the value.
     */
    val presentation: String?
    companion object {
        /**
         * Date and time field type. Used when defining rule requirements.
         */
        val dateTimeType: String?

        /**
         * Date field type. Used when defining rule requirements.
         */
        val dateType: String?

        /**
         * Float field type. Used when defining rule requirements.
         */
        val floatType: String?

        /**
         * Integer field type. Used when defining rule requirements.
         */
        val integerType: String?

        /**
         * Period field type. Used when defining rule requirements.
         */
        val periodType: String?

        /**
         * String field type. Used when defining rule requirements.
         */
        val stringType: String?

        /**
         * Text field type. Used when defining rule requirements.
         */
        val textType: String?
    }
}

/**
 * Represents a custom field in a project that stores a UserGroup type.
 */
open external class GroupProjectCustomField : ProjectCustomField {
    /**
     * The set of default values.
     */
    var defaultValues: Iterable<UserGroup>?

    /**
     * The set of available values for this custom field.
     */
    val values: Iterable<UserGroup>?

    /**
     * Returns the value that matches the specified name in a custom field that stores values as a user group.
     * @param name The name of the group to search for in the set of values for the custom field.
     * @return The group with the specified name. This group can be set as the value for a field that stores a user group.
     */
    fun findValueByName(name: String?): UserGroup?
}

/**
 * Represents an issue in YouTrack.
 */
open external class Issue (
        reporter: User?,
        project: Project?,
        summary: String?
) : BaseEntity {
    /**
     * The set of attachments that are attached to the issue.
     */
    val attachments: Iterable<IssueAttachment>?

    /**
     * A list of comments for the issue.
     */
    val comments: Iterable<IssueComment>?

    /**
     * The date when the issue was created.
     */
    val created: Long?

    /**
     * The text that is entered as the issue description.
     */
    var description: String?

    /**
     * When `true`, the issue description is parsed as Markdown. When `false`, the issue description is parsed as YouTrack Wiki.
     * Changing this value does not transform the markup from one syntax to another.
     *
     * @since 2017.4.38870
     */
    var isUsingMarkdown: Boolean?

    /**
     * The issue number in the project.
     */
    val numberInProject: Long?

    /**
     * The user group for which the issue is visible. If the property contains a null value, the issue is visible to the All Users group.
     */
    var permittedGroup: UserGroup?

    /**
     * The groups for which the issue is visible when the visibility is restricted to multiple groups.
     */
    var permittedGroups: Iterable<UserGroup>?

    /**
     * The list of users for whom the issue is visible.
     */
    var permittedUsers: Iterable<User>?

    /**
     * The project to which the issue is assigned.
     */
    var project: Project?

    /**
     * The user who reported (created) the issue.
     * @example
     * issue.fields.Assignee = issue.reporter;
     */
    val reporter: User

    /**
     * The date and time when the issue was assigned a state that is considered to be resolved.
     */
    val resolved: Long?

    /**
     * The collection of sprints that this issue has been added to.
     */
    var sprints: Iterable<Sprint>?

    /**
     * The text that is entered as the issue summary.
     */
    var summary: String?

    /**
     * The list of tags that are attached to an issue.
     */
    var tags: Iterable<IssueTag>?

    /**
     * The date when the issue was last updated.
     */
    val updated: Long?

    /**
     * The user who last updated the issue.
     */
    val updatedBy: User

    /**
     * The number of votes for an issue. For vote-related methods, see User.canVoteIssue, User.voteIssue, User.canUnvoteIssue, and User.unvoteIssue.
     */
    val votes: Int?

    /**
     * If the issue becomes reported in the current transaction, this property is `true`.
     * @example
     * if (issue.fields.Subsystem !== null && issue.fields.Assignee === null &&
     *     (((issue.isChanged('Subsystem') || issue.isChanged('project') && issue.isReported) ||
     *     issue.becomesReported) {
     *   issue.fields.Assignee = issue.fields.Subsystem.owner
     * }
     */
    val becomesReported: Boolean?

    /**
     * If the issue is assigned a state that is considered resolved in the current transaction, this property is `true`.
     */
    val becomesResolved: Boolean?

    /**
     * If the issue is assigned a state that is considered unresolved in the current transaction. this property is `true`.
     */
    val becomesUnresolved: Boolean?

    /**
     * The root issue in a tree of duplicates that are linked to the issue.
     * For example, if `issueA` duplicates `issueB` and `issueB` duplicates
     * `issueC`, then the value for the `issueA.duplicateRoot()` property is `issueC`.
     */
    val duplicateRoot: Issue?

    /**
     * The set of comments that are edited in the current transaction.
     * Comments that are added and removed are not considered to be edited.
     * Instead, these are represented by the `issue.comments.added` and
     * `issue.comments.removed` properties.
     */
    val editedComments: Iterable<IssueComment>?

    /**
     * The set of work items that are edited in the current transaction.
     * Work items that are added and removed are not considered to be edited.
     * Instead, these are represented by the `issue.workItems.added` and
     * `issue.workItems.removed` properties.
     *
     * @since 2017.4.37824
     */
    val editedWorkItems: Iterable<IssueWorkItem>?

    /**
     * The visible ID of an issue or similar object in an originating third-party system.
     *
     * @since 2018.2.42312
     */
    val externalId: String?

    /**
     * The URL for an issue or similar object in an originating third-party system.
     *
     * @since 2018.2.42312
     */
    val externalUrl: String?

    /**
     * The issue ID.
     * @example
     * user.notify('Issue is overdue', 'Please, look at the issue: ' + issue.id);
     */
    val id: String?

    /**
     * The absolute URL that points to the issue.
     * @example
     * user.notify('Issue is overdue', 'Please, look at the issue: ' + issue.url);
     */
    val url: String?

    /**
     * The list of VCS changes that are associated with the issue.
     *
     * @since 2018.1.38923
     */
    val vcsChanges: Iterable<VcsChange>?

    /**
     * The set of work items that have been added to the issue.
     */
    val workItems: Iterable<IssueWorkItem>?

    /**
     * If the issue is already reported or becomes reported in the current transaction, this property is `true`. To apply changes to an issue draft, use `!issue.isReported`.
     * @example
     * issue.links['depends on'].forEach(function(dep) {
     *   if (dep.isReported) {
     *     assert(dep.State.resolved, 'The issue has unresolved dependencies and thus cannot be set Fixed!');
     *   }
     * });
     */
    val isReported: Boolean?

    /**
     * If the issue is currently assigned a state that is considered resolved, this property is `true`.
     */
    val isResolved: Boolean?

    /**
     * If the current user has added the star tag to watch the issue, this property is `true`.
     */
    val isStarred: Boolean?

    /**
     * __ignore
     * @param text TODO document
     */
    fun addComment(text: String?): IssueComment?

    /**
     * Adds a comment to the issue.
     * Makes `issue.comments.isChanged` return `true` for the current transaction.
     * @param text The text to add to the issue as a comment.
     * @param author The author of the comment.
     * @return A newly created comment.
     */
    fun addComment(text: String?, author: User?): IssueComment?

    /**
     * Adds a tag with the specified name to an issue. YouTrack adds the first matching tag that is visible to the current user.
     * If a match is not found, a new private tag is created for the current user.
     * When you use this method to add the star tag to an issue, the current user is added to the list of watchers.
     * To add a tag that is not visible to the current user, use the applyCommand method instead.
     * Use "add tag [tagName]" for the command parameter and specify the login for the owner of the tag in the runAs parameter.
     * @example
     * Issue.addTag('doit');
     * @param name The name of the tag to add to the issue.
     * @return The tag that has been added to the issue.
     */
    fun addTag(name: String?): IssueTag?

    /**
     * Adds a work item to the issue.
     * @param description The description of the work item.
     * @param date The date that is assigned to the work item.
     * @param author The user who performed the work.
     * @param duration The work duration in minutes.
     * @param type The work item type.
     * @return The new work item.
     */
    fun addWorkItem(
            description: String?,
            date: Long?,
            author: User?,
            duration: Int?,
            type: WorkItemType?
    ): IssueWorkItem?

    /**
     * __ignore
     * @param command TODO document
     */
    fun applyCommand(command: String?): Unit?

    /**
     * Applies a command to the issue.
     * @param command The command that is applied to the issue.
     * @param runAs Specifies the user by which the command is applied. If this parameter is not set, the command is applied on behalf of the current user.
     */
    fun applyCommand(command: String?, runAs: User?): Unit?

    /**
     * Removes all of the attachments from the issue.
     */
    fun clearAttachments(): Unit?

    /**
     * Creates a copy of the issue.
     * @param project Project to create new issue in. Available since 2018.1.40575.
     * @return The copy of the original issue.
     */
    fun copy(project: Project?): Issue?

    /**
     * __ignore
     * @return TODO documentation missing
     */
    fun copy(): Issue?

    /**
     * Creates a copy of the issue. No workflows will be triggered for the newly created issue.
     * @Deprecated use copy() instead.
     * @return The copy of the original issue.
     */
    fun createCopy(): Issue?

    /**
     * Checks whether the specified tag is attached to an issue.
     * @param tagName The name of the tag to check for the issue.
     * @return If the specified tag is attached to the issue, returns `true`.
     */
    fun hasTag(tagName: String?): Boolean?

    /**
     * Checks whether the issue is accessible by specified user.
     * @param user The user to check.
     * @return If the issue is accessbie for the user, returns 'true'.
     */
    fun isVisibleTo(user: User?): Boolean?

    /**
     * Removes a tag with the specified name from an issue. If the specified tag is not attached to the issue, an exception is thrown.
     * This method first searches through tags owned by the current user, then through all other visible tags.
     * To remove a tag that is not visible to the current user, use the applyCommand method instead.
     * Use "remove tag [tagName]" for the command parameter and specify the login for the owner of the tag in the runAs parameter.
     * @example
     * Issue.removeTag('waiting for reply');
     * @param name The name of the tag to remove from the issue.
     * @return The tag that has been removed from the issue.
     */
    fun removeTag(name: String?): IssueTag?

    /**
     * __ignore
     * @param text  
     */
    fun wikify(text: String?): String?

    /**
     * Converts text with Markdown or YouTrack Wiki markup to HTML. Use this method to send "pretty" notifications.
     * @example
     * issue.Assignee.notify('Comment added:', issue.wikify(comment.text, comment.isUsingMarkdown));
     * @param text The string of text to convert to HTML.
     * @param usingMarkdown If `true`, the markup is parsed as Markdown. If `false`, the markup is parsed as YouTrack Wiki. If omitted, issue.isUsingMarkdown is used implicitly. Available since 2018.1.40100.
     * @return The wikified string.
     */
    fun wikify(text: String?, usingMarkdown: Boolean?): String?
    companion object {
        /**
         * Finds an issue by its visible ID.
         * @param id The issue ID.
         * @return The issue that is assigned the specified ID.
         */
        fun findById(id: String?): Issue?

        /**
         * __ignore
         * @param folder  
         * @param query  
         */
        fun search(folder: WatchFolder?, query: String?): Iterable<Issue>?

        /**
         * __ignore
         * @param folder  
         * @param query  
         * @param user  
         */
        fun search(
                folder: WatchFolder?,
                query: String?,
                user: User?
        ): Iterable<Issue>?
    }
}

/**
 * Represents a file that is attached to an issue.
 */
open external class IssueAttachment : PersistentFile {
    /**
     * The user who attached the file to the issue.
     */
    val author: User?

    /**
     * The date and time when the attachment was created as a timestamp.
     */
    val created: Long?

    /**
     * If the attachment is removed, this property is `true`.
     */
    val isRemoved: Boolean?

    /**
     * The issue that the file is attached to.
     */
    val issue: Issue

    /**
     * The dimentions of an image file. For an image file, the value is
     * `rw=_width_&rh=_heigth_`, for a non-image file the value is `empty`.
     */
    val metaData: String?

    /**
     * The group for which the attachment is visible when the visibility is restricted to a single group.
     */
    var permittedGroup: UserGroup?

    /**
     * The groups for which the issue is visible when the visibility is restricted to multiple groups.
     */
    var permittedGroups: Iterable<UserGroup>?

    /**
     * The list of users for whom the attachment is visible.
     */
    var permittedUsers: Iterable<User>?

    /**
     * The date and time the attachment was last updated as a timestamp.
     */
    val updated: Long?

    /**
     * Permanently deletes the attachment.
     *
     * @since 2018.1.40030
     */
    fun delete(): Unit?
}

/**
 * Represents a comment that is added to an issue.
 */
open external class IssueComment : BaseComment {
    /**
     * The user who created the comment.
     */
    val author: User

    /**
     * Time the comment was created.
     */
    val created: Long?

    /**
     * The issue the comment belongs to.
     */
    val issue: Issue?

    /**
     * A group whos members are allowed to access the comment.
     */
    var permittedGroup: UserGroup?

    /**
     * Groups whos members are allowed to access the comment.
     */
    var permittedGroups: Iterable<UserGroup>?

    /**
     * Users that are allowed to access the comment.
     */
    var permittedUsers: Iterable<User>?

    /**
     * The text of the comment.
     */
    var text: String?

    /**
     * Time the comment was last updated.
     */
    val updated: Long?

    /**
     * The user who last updated the comment.
     */
    val updatedBy: User?

    /**
     * The absolute URL (permalink) that points to the comment.
     * @example
     * user.notify('Somebody has written something', 'Have a look: ' + comment.url);
     */
    val url: String?

    /**
     * Logically deletes the comment. This means that the comment is marked as deleted, but remains in the database.
     * Users with sufficient permissions can restore the comment or delete the comment permanently from the user interface.
     * The option to delete comments permanently has not been implemented in this API.
     *
     * @since 2018.1.38923
     */
    fun delete(): Unit?
}

/**
 * Represents an issue link type.
 */
open external class IssueLinkPrototype : BaseEntity {
    /**
     * The inward name of the issue link type.
     */
    val inward: String?

    /**
     * The outward name of the issue link type.
     */
    val outward: String?
    companion object {
        /**
         * Finds an issue link type by its inward or outward name.
         * @param name The inward or outward name of an issue link type.
         * @return The issue link type.
         */
        fun findByName(name: String?): IssueLinkPrototype?
    }
}

/**
 * Represents a tag.
 */
open external class IssueTag : WatchFolder {
    /**
     * The user who created the tag.
     */
    val owner: User
    companion object {
        /**
         * Finds a list of tags with the specified name. The list only includes tags that are visible to the current user.
         * The tags that were created by the current user are returned at the top of the list.
         * "Star" tag is excluded from the results.
         * @param name The name of the tag to search for.
         * @return A list of tags that match the specified name.
         */
        fun findByName(name: String?): Iterable<IssueTag>?

        /**
         * Finds the most relevant tag with the specified name that is visible to the current user.
         * "Star" tag is excluded from the results.
         * @param name The name of the tag to search for.
         * @return The most relevant tag.
         */
        fun findTagByName(name: String?): IssueTag?
    }
}

/**
 * Represents a work item that has been added to an issue.
 */
open external class IssueWorkItem : BaseWorkItem {
    /**
     * The date and time that is assigned to the work item. Stored as a Unix timestamp in UTC. The time part is set to midnight for the current date.
     */
    val date: Long

    /**
     * The duration of the work item in minutes.
     * Writable since 2018.1.40800
     */
    var duration: Int?

    /**
     * Permanently deletes the work item.
     *
     * @since 2018.2.42312
     */
    fun delete(): Unit?
}

/**
 * Represents a value in a custom field that has a user associated with it, a so-called owner.
 */
open external class OwnedField : Field {
    /**
     * The user who is associated with the value.
     */
    val owner: User?
    companion object {
        /**
         * Field type. Used when defining rule requirements.
         */
        val fieldType: String?
    }
}

/**
 * Represents a custom field in a project that stores a value as a period type.
 * We use org.joda.time.Period as a base class for period values.
 * While you can read the class documentation at
 * http://joda-time.sourceforge.net/apidocs/org/joda/time/Period.html,
 * please note that we support only class members which use the Period class and
 * primitive types like String and int.
 * @example
 * // to convert period to minutes (or other units) use get* methods:
 * var period = issue.fields.Estimation;
 * var minutes = !period ? 0 : (period.getMinutes() +
 *                              60 * (period.getHours() +
 *                                    8 * period.getDays()));
 * // to create Period instance, use toPeriod function from date-time module:
 * issue.fields.Estimation = dateTime.toPeriod(3 * 3600 * 1000); // 3h in ms
 * issue.fields.Estimation = dateTime.toPeriod('3h'); // short form
 * issue.fields.Estimation = dateTime.toPeriod('2w4d3h15m'); // full form
 */
open external class PeriodProjectCustomField : SimpleProjectCustomField

/**
 * TODO document
 */
open external class PersistentFile : BaseEntity {
    /**
     * The extension that defines the file type.
     */
    val extension: String?

    /**
     * The name of the file.
     */
    val name: String?

    /**
     * The size of the attached file in bytes.
     */
    val size: Long?
}

/**
 * Represents a YouTrack project.
 */
open external class Project : BaseEntity {
    /**
     * The description of the project as shown on the project profile page.
     */
    val description: String?

    /**
     * The set of custom fields that are available in the project.
     */
    val fields: Iterable<ProjectCustomField>?

    /**
     * A list of all issues that belong to the project.
     */
    val issues: Iterable<Issue>?

    /**
     * The user who is set as the project lead.
     */
    val leader: User

    /**
     * The ID of the project. Use instead of project.shortName, which is deprecated.
     */
    val key: String?

    /**
     * The name of the project.
     */
    val name: String?

    /**
     * The email address that is used to send notifications for the project.
     * If a 'From' address is not set for the project, the default 'From' address for the YouTrack server is returned.
     * @example
     * if (issue.becomesReported) {
     *   var lastRelatedEmails = issue.fields['Last message related emails'];
     *   if (lastRelatedEmails) {
     *     lastRelatedEmails.split(' ').forEach(function (email) {
     *       if (email && email.equalsIgnoreCase(issue.project.notificationEmail)) {
     *         var allRelatedEmails = issue.fields['All related emails'];
     *         if (!allRelatedEmails) {
     *           issue.fields['All related emails'] = email;
     *         } else if (!(allRelatedEmails.split(' ').has(email))) {
     *           issue.fields['All related emails'] = allRelatedEmails + ' ' + email;
     *         }
     *       }
     *     });
     *     issue.fields['Last message related emails'] = null;
     *   }
     * }
     */
    val notificationEmail: String?

    /**
     * A UserGroup object that contains the users and members of groups who are assigned to the project team.
     *
     * @since 2017.4.38235
     */
    val team: UserGroup?

    /**
     * If the project is currently archived, this property is `true`.
     */
    val isArchived: Boolean?

    /**
     * Returns the custom field in the project with the specified name.
     * @param name The name of the custom field.
     * @return The custom field with the specified name.
     */
    fun findFieldByName(name: String?): ProjectCustomField?

    /**
     * Gets the number of minutes that occurred during working hours in a specified interval.
     * For example, if the interval is two days and the number of working hours in a day is set to 8, the result is 2 * 8 * 60 = 960
     * @param start Start of the interval.
     * @param end End of the interval.
     * @return The number of minutes that occurred during working hours in the specified interval.
     */
    fun intervalToWorkingMinutes(start: Long?, end: Long?): Long?
    companion object {
        /**
         * Finds a project by ID.
         * @param key The ID of the project to search for.
         * @return The project, or null when there are no projects with the specified ID.
         */
        fun findByKey(key: String?): Project?

        /**
         * Finds a project by name.
         * @param name The name of the project to search for.
         * @return The project, or null when there are no projects with the specified name.
         */
        fun findByName(name: String?): Project?
    }
}

/**
 * Represents a custom field that is available in a project.
 */
open external class ProjectCustomField : BaseEntity {
    /**
     * The text that is displayed for this field when it is empty.
     */
    val nullValueText: String?

    /**
     * The localized name of the field.
     */
    val localizedName: String?

    /**
     * The name of the field.
     */
    val name: String?

    /**
     * Checks if the changes that are applied in the current transaction remove the condition to show the custom field.
     * @param issue The issue for which the condition for showing the field is checked.
     * @return When `true`, the condition for showing the field is removed in the current transaction.
     *
     * @since 2018.2.42312
     */
    fun becomesInvisibleInIssue(issue: Issue?): Boolean?

    /**
     * Checks if the changes that are applied in the current transaction satisfy the condition to show the custom field.
     * @param issue The issue for which the condition for showing the field is checked.
     * @return When `true`, the condition for showing the field is met in the current transaction.
     *
     * @since 2018.2.42312
     */
    fun becomesVisibleInIssue(issue: Issue?): Boolean?

    /**
     * Returns the background color that is used for this field value in the specified issue.
     * Can return `null`, `"white"`, or a hex color presentation.
     * @param issue The issue for which the background color is returned.
     * @return The background color that is used for this field value in the specified issue.
     */
    fun getBackgroundColor(issue: Issue?): String?

    /**
     * Returns the foreground color that is used for this field value in the specified issue.
     * Can return `null`, `"white"`, or a hex color presentation.
     * @param issue The issue for which the foreground color is returned.
     * @return The foreground color that is used for this field value in the specified issue.
     */
    fun getForegroundColor(issue: Issue?): String?

    /**
     * Returns the string presentation of the value that is stored in this field in the specified issue.
     * @param issue The issue for which the value presentation is returned.
     * @return The string presentation of the value.
     */
    fun getValuePresentation(issue: Issue?): String?

    /**
     * Checks if a field is visible in the issue.
     * @param issue The issue for which the condition for showing the field is checked.
     * @return When `true`, the condition for showing the custom field in the issue has been met. It can also mean that the field is not shown on a conditional basis and is always visible.
     *
     * @since 2018.2.42312
     */
    fun isVisibleInIssue(issue: Issue?): Boolean?
}

/**
 * Represents a value in a custom field that stores a version type.
 */
open external class ProjectVersion : Field {
    /**
     * The release date that is associated with the version.
     */
    val releaseDate: Long?

    /**
     * If the version is archived, this property is `true`.
     */
    val isArchived: Boolean?

    /**
     * If the version is released, this property is `true`.
     */
    val isReleased: Boolean?
    companion object {
        /**
         * Field type. Used when defining rule requirements.
         */
        val fieldType: String?
    }
}

/**
 * Represents a saved search.
 */
open external class SavedQuery : WatchFolder {
    /**
     * The search query.
     */
    val query: String?

    /**
     * The user who created the saved search.
     */
    val owner: User?
    companion object {
        /**
         * Finds a list of saved searches with the specified name. The list only includes saved searches that are visible to the current user.
         * The saved searches that were created by the current user are returned at the top of the list.
         * @param name The name of the saved search to search for.
         * @return A list of saved searches that match the specified name.
         */
        fun findByName(name: String?): Iterable<SavedQuery>?

        /**
         * Finds the most relevant saved search with the specified name that is visible to the current user.
         * @param name The name of the saved search to search for.
         * @return The most relevant saved search.
         */
        fun findQueryByName(name: String?): SavedQuery?
    }
}

/**
 * Base class for custom fields that store simple values like strings and numbers.
 */
open external class SimpleProjectCustomField : ProjectCustomField

/**
 * Represents a sprint that is associated with an agile board. Each sprint can include issues from one or more projects.
 */
open external class Sprint : BaseEntity {
    /**
     * The agile board that the sprint belongs to.
     */
    val agile: Agile?

    /**
     * The end date of the sprint.
     */
    val finish: Long?

    /**
     * The start date of the sprint.
     */
    val start: Long?

    /**
     * The name of the sprint.
     */
    val name: String?

    /**
     * If the sprint is currently archived, this property is `true`.
     */
    val isArchived: Boolean?

    /**
     * Checks whether the specified issue is represented as a swimlane on the agile board that the sprint belongs to.
     * @param issue The issue to check.
     * @return If the specified issue is represented as a swimlane in the sprint, returns `true`.
     */
    fun isSwimlane(issue: Issue?): Boolean?
}

/**
 * Represents a value in a custom field that stores a state type.
 */
open external class State : Field {
    /**
     * If issues in this state are considered to be resolved, ths property is `true`.
     */
    val isResolved: Boolean?
    companion object {
        /**
         * Field type. Used when defining rule requirements.
         */
        val fieldType: String?
    }
}

/**
 * Represents a user account in YouTrack.
 */
open external class User : BaseEntity {
    /**
     * The email address of the user.
     */
    val email: String?

    /**
     * The full name of the user as seen in their profile.
     */
    val fullName: String?

    /**
     * The login of the user.
     */
    val login: String?

    /**
     * User's time zone id.
     */
    val timeZoneId: String?

    /**
     * The full name of the user or the login if the full name is not set.
     */
    val visibleName: String?

    /**
     * If the user is currently banned, this property is `true`.
     */
    val isBanned: Boolean?

    /**
     * Checks whether the user is able to remove their vote from the specified issue.
     * @param issue The issue to check.
     * @return If the user can vote for the issue, returns `true`.
     */
    fun canUnvoteIssue(issue: Issue?): Boolean?

    /**
     * Checks whether the user is able to vote for the specified issue.
     * @param issue The issue to check.
     * @return If the user can vote for the issue, returns `true`.
     */
    fun canVoteIssue(issue: Issue?): Boolean?

    /**
     * Returns a tag with the specified name that is shared with but not owned by the user. If such a tag does not exist, a null value is returned.
     * @param name The name of the tag.
     * @return The tag.
     */
    fun getSharedTag(name: String?): IssueTag?

    /**
     * Returns a tag that is visible to the user.
     * @param name The name of the tag.
     * @param createIfNotExists If `true` and the specified tag does not exist or is not visible to the user and the user has permission to create tags, a new tag with the specified name is created.
     * @return The tag.
     */
    fun getTag(name: String?, createIfNotExists: Boolean?): IssueTag?

    /**
     * __ignore
     * @param roleName TODO document
     */
    fun hasRole(roleName: String?): Boolean?

    /**
     * Checks whether the user is granted the specified role in the specified project. When the project parameter is not specified, checks whether the user has the specified role in any project.
     * @param roleName The name of the role to check for.
     * @param project The project to check for the specified role assignment. If omitted, checks if the user has the global role.
     * @return If the user is granted the specified role, returns `true`.
     */
    fun hasRole(roleName: String?, project: Project?): Boolean?

    /**
     * Checks whether the user is a member of the specified group.
     * @param groupName The name of the group to check for.
     * @return If the user is a member of the specified group, returns `true`.
     */
    fun isInGroup(groupName: String?): Boolean?

    /**
     * __ignore
     * @param subject TODO document
     * @param body TODO document
     */
    fun notify(subject: String?, body: String?): Unit?

    /**
     * __ignore
     * @param subject TODO document
     * @param body TODO document
     * @param ignoreNotifyOnOwnChangesSetting TODO document
     */
    fun notify(
            subject: String?,
            body: String?,
            ignoreNotifyOnOwnChangesSetting: Boolean?
    ): Unit?

    /**
     * Sends an email notification to the email address that is set in the user profile.
     * @param subject The subject line of the email notification.
     * @param body The message text of the email notification.
     * @param ignoreNotifyOnOwnChangesSetting If `false`, the message is not sent when changes are performed on behalf of the current user. Otherwise, the message is sent anyway.
     * @param project When set, the email address that is used as the 'From' address for the specified project is used to send the message.
     */
    fun notify(
            subject: String?,
            body: String?,
            ignoreNotifyOnOwnChangesSetting: Boolean?,
            project: Project?
    ): Unit?

    /**
     * Sends a notification message over Jabber. Similar to the `notify`
     * method, the message won't be sent on own changes and corresponding flag unchecked.
     * @param message The message text for the Jabber notification.
     */
    fun sendJabber(message: String?): Unit?

    /**
     * Sends an email notification to the email address that is set in the user profile. An alias for notify(subject, body, true).
     * @param subject The subject line of the email notification.
     * @param body The message text of the email notification.
     */
    fun sendMail(subject: String?, body: String?): Unit?

    /**
     * Removes a vote on behalf of the user from the issue, if allowed.
     * @param issue The issue from which the vote is removed.
     */
    fun unvoteIssue(issue: Issue?): Unit?

    /**
     * Removes the current user from the list of watchers for the issue
     * (remove `Star` tag).
     * @param issue The issue to from which the user is removed as a watcher.
     */
    fun unwatchIssue(issue: Issue?): Unit?

    /**
     * Adds a vote on behalf of the user to the issue, if allowed.
     * @param issue The issue to which the vote is added.
     */
    fun voteIssue(issue: Issue?): Unit?

    /**
     * Adds the current user to the issue as a watcher (add `Star` tag).
     * @param issue The issue to which the user is added as a watcher.
     */
    fun watchIssue(issue: Issue?): Unit?
    companion object {
        /**
         * The current (logged in) user.
         */
        val current: User?

        /**
         * Field type. Used when defining rule requirements.
         */
        val fieldType: String?

        /**
         * Finds users by email.
         * @param email The email to search for.
         * @return Users with the specified email.
         *
         * @since 2018.2.41100
         */
        fun findByEmail(email: String?): Iterable<User>?

        /**
         * Finds a user by login.
         * @param login The login of the user account to search for.
         * @return The specified user, or null when a user with the specified login is not found.
         */
        fun findByLogin(login: String?): User?

        /**
         * Finds a user by email.
         * @param email The email of the user account to search for.
         * @return The specified user, or null when a user with the specified email is not found or there are multiple users with the specified email.
         *
         * @since 2018.2.41100
         */
        fun findUniqueByEmail(email: String?): User?
    }
}

/**
 * Represents a group of users.
 */
open external class UserGroup : BaseEntity {
    /**
     * The description of the group.
     */
    val description: String?

    /**
     * The name of the group.
     */
    val name: String?

    /**
     * A list of users who are members of the group.
     */
    val users: Iterable<User>?

    /**
     * If the group is the All Users group, this property is `true`.
     */
    val isAllUsersGroup: Boolean?

    /**
     * If the auto-join option is enabled for the group, this property is `true`.
     */
    val isAutoJoin: Boolean?

    /**
     * Sends an email notification to all of the users who are members of the group.
     * @example
     * issue.oldValue('permittedGroup').notifyAllUsers('Visibility has been changed',
     *   'The visibility group for the issue ' + issue.getId() +
     *   ' has been changed to ' + permittedGroup.name);
     * @param subject The subject line of the email notification.
     * @param body The message text of the email notification.
     */
    fun notifyAllUsers(subject: String?, body: String?): Unit?
    companion object {
        /**
         * The All Users group.
         */
        val allUsersGroup: UserGroup?

        /**
         * Field type. Used when defining rule requirements.
         */
        val fieldType: String?

        /**
         * Finds a group by name.
         * @param name The name of the group to search for.
         * @return The specified user group, or null when a group with the specified name is not found.
         */
        fun findByName(name: String?): UserGroup?
    }
}

/**
 * Represents a custom field in a project that stores values as a user type.
 */
open external class UserProjectCustomField : ProjectCustomField {
    /**
     * The default value for the custom field.
     */
    val defaultUsers: Iterable<User>?

    /**
     * The list of available values for the custom field.
     */
    val values: Iterable<User>?

    /**
     * Returns the value that matches the specified login in a custom field that stores values as a user type.
     * @param login The user login to search for in the set of values for the custom field.
     * @return The user with the specified login. This user can be set as the value for a field that stores a user type.
     */
    fun findValueByLogin(login: String?): User?
}

/**
 * Represents a VCS change that is attached to an issue.
 */
open external class VcsChange : BaseEntity {
    /**
     * The name of the branch that the VCS change was committed to.
     *
     * @since 2018.1.38923
     */
    val branch: String?

    /**
     * The date when the change was applied, as returned by the VCS.
     * Deprecated from 2018.1.39547. Use `VcsChange.created` instead.
     *
     * @since 2018.1.38923
     */
    val date: Long?

    /**
     * The date when the VCS change was retrieved from the change processor.
     *
     * @since 2018.1.39547
     */
    val fetched: Long?

    /**
     * A unique identifier. Used by some CI servers in addition to version.
     *
     * @since 2018.1.38923
     */
    val id: Long?

    /**
     * The user who authored the VCS change.
     *
     * @since 2018.1.38923
     */
    val user: User?

    /**
     * The name of the change author, as returned by the VCS.
     *
     * @since 2018.1.38923
     */
    val userName: String?

    /**
     * The version number of the change. For a Git-based VCS, the revision hash.
     *
     * @since 2018.1.38923
     */
    val version: String?

    /**
     * The list of change processors that the VCS change can be retrieved from.
     *
     * @since 2018.1.38923
     */
    val changesProcessors: Iterable<ChangesProcessor>?

    /**
     * The date when the change was applied, as returned by the VCS.
     *
     * @since 2018.1.39547
     */
    val created: Long?

    /**
     * The comment text that was included in the commit message.
     *
     * @since 2018.1.38923
     */
    val text: String?
}

/**
 * Represents a VCS server.
 */
open external class VcsServer : BaseEntity {
    /**
     * The URL of the VCS server.
     *
     * @since 2018.1.38923
     */
    val url: String
}

/**
 * Represents a common ancestor of classes that represent tags and saved searches.
 */
open external class WatchFolder : BaseEntity {
    /**
     * The name of the tag or saved search.
     */
    val name: String?

    /**
     * The group of users for whom the tag or saved search is visible.
     * If the tag or the saved search is only visible to its owner, the value for this property is `null`.
     */
    val shareGroup: UserGroup?

    /**
     * The group of users who are allowed to update the settings for the tag or saved search.
     * If the tag or the saved search can only be updated by its owner, the value for this property is `null`.
     */
    val updateShareGroup: UserGroup?
}

/**
 * Represents a work type that can be assigned to a work item.
 */
open external class WorkItemType : BaseEntity {
    /**
     * The name of the work item type.
     */
    val name: String?
    companion object {
        /**
         * Returns the set of work item types that are available in a project.
         * @param project The project for which work item types are returned.
         * @return The set of available work item types for the specified project.
         */
        fun findByProject(project: Project?): Iterable<WorkItemType>?
    }
}
abstract external class BaseEntity {

    /**
     * When `true`, the entity is removed in the current transaction. Otherwise, `false`.
     * @since 2017.4.37915
     */
    val becomesRemoved: Boolean

    /**
     * When `true`, the entity is created in the current transaction. Otherwise, `false`.
     * @since 2018.2.42351
     */
    val isNew: Boolean

    internal fun required(fieldName: String, message: String)

    internal fun becomes(fieldName: String, expected: Any?): Boolean

    internal fun isChanged(fieldName: String): Boolean

    internal fun oldValue(fieldName: String): Any?

    internal fun canBeReadBy(fieldName: String, user: User?): Boolean

    internal fun canBeWrittenBy(fieldName: String, user: User?): Boolean
}