/*
 Copyright 2017 JetBrains s.r.o.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

/**
 * This module contains utility functions that display workflow-related messages and warnings in the user interface.
 *
 * @module @jetbrains/youtrack-scripting-api/workflow
 */

var env = require('./environment');

/**
 * Displays the specified message in the YouTrack user interface.
 *
 * @param {string} [messageText] The message text.
 */
exports.message = function (messageText) {
  env.message(messageText);
};

/**
 * Checks to determine whether the specified condition is true.
 * If the condition is not true, the system throws an error.
 * All changes are rolled back to the initial state..
 * The error contains the specified message, which is displayed in the user interface.
 *
 * @example
 * workflow.check(issue.fields.Assignee && ctx.currentUser.login === issue.fields.Assignee.login,
 *   'Only Assignee can mark issue as Fixed.');
 *
 * @param {boolean} [condition] The condition to check.
 * @param {string} [message] The error message that is shown to the user in cases where the condition is false.
 */
exports.check = function (condition, message) {
  env.checkCondition(!!condition, message);
};

/**
 * Returns a localized version of a message.
 * WARNING: for internal use only!
 * This method is only supported in default workflows where the IDs and text strings for localized messages are stored in the application.
 * References to this method in custom workflows are not supported.
 *
 * @param {string} [messageId] The message ID.
 *
 * @returns {Object} An object that contains the placeholders and localized strings that are associated with the specified message.
 * When this object is passed to a function that expects a string value, the toString method is called internally,
 * replacing the parameters with the localized string.
 */
exports.i18n = function (messageId, p1, p2, p3) {
  var params = [].slice.call(arguments);
  params.splice(0, 1);
  return env.localize(messageId, params);
};